//<copyright>
//
// Copyright (c) 1994,95
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>
//
// Note: this implementation is based on similar code for IrisGL,
// see copyright notice of original version below

//<file>
//
// File:        oglcontext.C - glyph for 3D drawing
//              implementation of glcontext for OpenGL
//
// Created:     10 Aug 94   Michael Pichler
//
// Changed:     11 Nov 96   Michael Pichler
//
// $Id: oglcontext.C,v 1.9 1997/02/25 17:03:58 mpichler Exp $
//
//</file>

/*
 * Copyright (c) 1991 Stanford University
 * Copyright (c) 1991 Silicon Graphics, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the names of
 * Stanford and Silicon Graphics may not be used in any advertising or
 * publicity relating to the software without the specific, prior written
 * permission of Stanford and Silicon Graphics.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD OR SILICON GRAPHICS BE LIABLE FOR
 * ANY SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND,
 * OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF 
 * LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 */


#include "gecontext.h"
#include "wtable.h"

#ifdef GE3DTEXT
#include <ge3d/ge3d.h>
#include <InterViews/font.h>
#include <IV-X11/xfont.h>
#endif

#include <InterViews/canvas.h>
#include <InterViews/display.h>
#include <InterViews/session.h>
#include <InterViews/style.h>
#include <InterViews/window.h>

#include <hyperg/utils/verbose.h>

#include <IV-X11/Xlib.h>
#include <IV-X11/xcanvas.h>
#include <IV-X11/xdisplay.h>
#include <IV-X11/xwindow.h>

/* to be not warned about redefinitions in glxproto.h */
#if defined(ALPHA_OSF1) || defined(ALPHA_GNU)
# include <InterViews/leave-scope.h>   /* Drawable etc. */
# include <IV-X11/Xundefs.h>           /* Display, Font */
/* to get rid of conflicts in X11/Xproto.h (included by glxproto.h) */
# undef Window
# undef Drawable
# undef Font
# undef Cursor
# undef Colormap
# include <GL/glxproto.h>    /* is included by glx.h and redefines Font etc. temporarily */
# include <IV-X11/Xdefs.h>   /* define Font to be XFont */
# include <GL/glx.h>
# include <GL/glu.h>
# include <IV-X11/Xundefs.h>  /* get back to IV's names */
# include <InterViews/enter-scope.h>
#else
# include <IV-X11/Xdefs.h>      /* Display, Font of X11 */
# include <GL/glx.h>
# include <GL/glu.h>
# include <IV-X11/Xundefs.h>  /* get back to IV's names */
#endif

#include <iostream.h>
#include <stdlib.h>


// class GLWindow for Implementation of class GEContext


class GLWindow: public Window
{
  public:
    GLWindow (Glyph*, Window* parent, int overlay);
    ~GLWindow ();

    virtual void place (Coord x, Coord y);
    virtual void bind ();
    virtual void repair ();

    virtual void setattributes ()
    { set_attributes (); }

    void activateOverlay (int clear);
    void deactivateOverlay ();

  private:
    Window* parent_;
    XWindow overlay_xwin_;
    int double_buffered_;
    int requestoverlay_;
    int directrendering_;
    GLXContext glxcon_;
};



GLWindow::GLWindow (Glyph* g, Window* parent, int overlay)
: Window(g)
{
  parent_ = parent;
  requestoverlay_ = overlay;
  overlay_xwin_ = 0;
  glxcon_ = 0;

  Display* disp = parent->display ();
  display (disp);

  Style* s = new Style (disp->style ());
  s->alias ("GLWindow");
  // use double buffering for GL window (unless single buffered required explicitly)
  double_buffered_ = !s->value_is_on ("single_buffered");  // default on
  s->attribute ("double_buffered", "off");  // otherwise InterViews causes errors!
  directrendering_ = !s->value_is_on ("indirectRendering");  // default: direct rendering
  DEBUGNL ("direct rendering: " << (directrendering_ ? "on" : "off"));
  style (s);
}


GLWindow::~GLWindow()
{
  WindowRep* w = rep();
//cerr << "glXDestroyContext (" << (void*) w->display_->rep()->display_ << ", " << (void*) glxcon_ << ")" << endl;
  glXDestroyContext (w->display_->rep()->display_, glxcon_);
}


// no GLWindow::draw


void GLWindow::place (Coord x, Coord y)
{
  DEBUGNL ("GLWindow::place");

  Window::place (x, y);
  WindowRep* wrep = rep ();
  Display* dis = wrep->display_;
  wrep->xpos_ = dis->to_pixels (wrep->left_);
  wrep->ypos_ = parent_->canvas ()->pheight () - dis->to_pixels (wrep->bottom_) 
                - canvas ()->pheight ();
  if (wrep->xwindow_ != WindowRep::unbound)
    XMoveWindow (dis->rep()->display_, wrep->xwindow_, wrep->xpos_, wrep->ypos_);

  DEBUGNL ("finished GLWindow::place");
}



void GLWindow::bind ()
{
  DEBUGNL ("beginning GLWindow::bind");

  if (requestoverlay_)
    cerr << "sorry: overlay windows not yet ported to open GL!" << endl;

  DisplayRep* drep = display ()->rep ();
  XDisplay* xdpy = drep->display_;

  // should do this on creation of context, not on each bind!!!
  int erb, evb;  // return values of glXQueryExtension
  if (!glXQueryExtension (xdpy, &erb, &evb))
  {
    cerr << "3D oglcontext. fatal: no glx extension on X-Display!" << endl;
    exit (-1);
  }

  Style* style = display ()->style ();
  style->alias ("GLWindow");
  int alphabitplanes = style->value_is_on ("alphaBitplanes");
  DEBUGNL ("alpha bitplanes: " << (alphabitplanes ? "on" : "off"));

  int init_config [32];  // request a double buffered, z-buffered RGB visual
  int* icfptr = init_config;
  if (double_buffered_)
    *icfptr++ = GLX_DOUBLEBUFFER;
  *icfptr++ = GLX_RGBA;
  *icfptr++ = GLX_RED_SIZE;
  *icfptr++ = 1;
  *icfptr++ = GLX_GREEN_SIZE;
  *icfptr++ = 1;
  *icfptr++ = GLX_BLUE_SIZE;
  *icfptr++ = 1;
  if (alphabitplanes)
  { *icfptr++ = GLX_ALPHA_SIZE;
    *icfptr++ = 1;
    // cerr << "requesting visual with alpha bitplanes" << endl;
  }
  *icfptr++ = GLX_DEPTH_SIZE;
  *icfptr++ = 1;
  *icfptr = (int) None;  // sentinel

  XVisualInfo* visual = glXChooseVisual (xdpy, DefaultScreen (xdpy), init_config);
  if (!visual)
  { cerr << "fatal: desired GLX visual (RGB with double- and z-buffer) not available!" << endl;
    exit (-1);
  }

  static GLXContext firstglxcontext = None;

  // direct flag (last one): True = use HW (if present), False = go through X server
  // direct rendering only possible to screen (not to pixmap)
  glxcon_ = glXCreateContext (
    xdpy, visual,
    firstglxcontext,  // share display lists with first glxcontext created
    directrendering_  // "direct rendering" flag
  );

//cerr << "glXCreateContext: " << (void*) glxcon_ << endl;

  if (!glxcon_)
  { cerr << "3D oglcontext. fatal: could not create glx-context!" << endl;
    exit (-1);
  }

  if (firstglxcontext == None)
    firstglxcontext = glxcon_;

  Visual* vi_sual = visual->visual;
  unsigned int dep_th = visual->depth;

  DEBUGNL ("visual used for OpenGL has ID 0x" << hex << vi_sual->visualid << dec);

#if defined (VERBOSE) && !defined (NOVERBOSE)
  // report visual class
  DEBUG ("class of OpenGL visual used: ");
  switch (vi_sual->c_class)
  {
    case DirectColor:  DEBUGNL ("Direct Colour Visual");  break;
    case GrayScale:    DEBUGNL ("Grey Scale Visual");     break;
    case PseudoColor:  DEBUGNL ("Pseudo Colour Visual");  break;
    case TrueColor:    DEBUGNL ("True Colour Visual");    break;
    default:           DEBUGNL ("Visual of other type than DirectColor/GrayScale/PseudoColor/TrueColor");
  }
#endif

  // must create separate colour map
  XColormap color_map = XCreateColormap (xdpy, RootWindow (xdpy, visual->screen), vi_sual, AllocNone);

  // get returned config entries!!!


  // create a window

  WindowRep* w = rep ();
  Canvas* c = w->canvas_;
  WindowTable* t = drep->wtable_;
  XWindow xw = w->xwindow_;

  if (xw != WindowRep::unbound)
    t->remove (xw);

  Window::set_attributes ();
  w->xattrmask_ &= ~CWDontPropagate;     // IV 3.1
  w->xattrs_.do_not_propagate_mask = 0;  // IV 3.1
  w->xattrmask_ |= CWColormap;
  w->xattrs_.colormap = color_map;

  // interested in structure and pointer events
  w->xattrs_.event_mask = ExposureMask | StructureNotifyMask |
    ButtonPressMask | ButtonReleaseMask |
    PointerMotionMask | PointerMotionHintMask;

  DEBUGNL ("creating X window ...");

  xw = XCreateWindow (
    xdpy, parent_->rep ()->xwindow_,
    w->xpos_, w->ypos_, c->pwidth (), c->pheight (), 0, // border width
    dep_th, w->xclass_, vi_sual, w->xattrmask_, &w->xattrs_
  );

  DEBUGNL ("... X window created");

  c->rep ()->xdrawable_ = xw;
  t->insert (xw, this);
  w->xwindow_ = xw;  // this X window will be used for GL
  w->xtoplevel_ = w->toplevel_->rep ()->xwindow_;


  // no equivalent for glxlink

  XMapRaised (drep->display_, xw);

  // like in gl-Xlib example program it may be better to set colours
  // of overlay planes here with XStoreColor (also possible with ge3d_MapColori)

  if (requestoverlay_)
  { // give overlay window its own colour map
    XSetWMColormapWindows (xdpy, parent_->rep ()->xwindow_, &overlay_xwin_, 1);
  }

  DEBUGNL ("finished GLWindow::bind");

} // GLWindow::bind



void GLWindow::repair ()
{
  WindowRep* wrep = rep ();
  Canvas* c = wrep->canvas_;
  CanvasRep* can = c->rep();

  if (can->damaged_)
  {
    XDisplay* xdpy = wrep->display_->rep ()->display_;

//XSync (xdpy, 0);  // noNEVERnoNOnever
//cerr << "glXMakeCurrent (" << (void*) xdpy << ", " << (void*) wrep->xwindow_ << ", " << (void*) glxcon_ << ")" << endl;

    if (!glXMakeCurrent (xdpy, wrep->xwindow_, glxcon_))
      cerr << "3D oglcontext. error: glXMakeCurrent did not succeed!" << endl;

    // some OpenGL libraries automatically reshape the viewport, others don't
//cerr << "viewport: " << flush;
//glXWaitGL ();  // crash caused by glXMakeCurrent on SGI; unknown reason (invalid window?)
//cerr << c->pwidth () << " x " << c->pheight () << endl;

    glViewport (0, 0, c->pwidth (), c->pheight ());

//cerr << "...done viewport" << endl;

/*
    // GL: in no double buffering is available, redraw only the damaged region
    if (!double_buffered_)
    { CanvasDamage& dam = can->damage_;
      Display* d = can->display_;
      scrmask (d->to_pixels (dam.left), d->to_pixels (dam.right),
               d->to_pixels (dam.bottom), d->to_pixels (dam.top));
    }
*/
    wrep->glyph_->draw (wrep->canvas_, wrep->allocation_);

    if (double_buffered_)
      glXSwapBuffers (xdpy, wrep->xwindow_);
    else
      glFlush ();

    can->clear_damage();
  }

//cerr << "GLWindow::repair () done" << endl;

} // repair


void GLWindow::activateOverlay (int /*clearing*/)
{
}


void GLWindow::deactivateOverlay ()
{
}


#ifdef GE3DTEXT
// ge3d_text

void ge3d_text (float x, float y, float z, const char* str)
{
  static int fontlistbase = -1;

//cerr << "writing '" << str << "' at position (" << x << ", " << y << ", " << z << ")." << endl;

  if (fontlistbase < 0)
  {
#ifdef OLDFONTIMPL
    const Font* font = Font::lookup ("fixed");
    if (!font)
    { fontlistbase = 0;
      return;
    }
    Resource::ref (font);
    FontRep* rep = font->rep (Session::instance ()->default_display ());
    // Interviews change: FontRep now XFontSet instead of XFontStruct

    long fid = rep->font_->fid;
#else
    XDisplay* dpy = Session::instance ()->default_display ()->rep()->display_;
    XFontStruct* xf = XLoadQueryFont (dpy, "fixed");

    if (!xf)
    { cerr << "VRweb: unable to load fixed font" << endl;
      fontlistbase = 0;
      return;
    }

    long fid = xf->fid;
#endif

    // supported characters: ' ' (0x20) to DEL (0x7f), i.e. 96 chars
    fontlistbase = glGenLists (96);
    if (fontlistbase)
      glXUseXFont (fid, 0x20, 96, fontlistbase);

#ifdef OLDFONTIMPL
    Resource::unref (font);
#endif
  }

  if (fontlistbase)  // assuming correct characters
  {
    const char* chr = str;
    int base2 = fontlistbase - 0x20;

    glRasterPos3f (x, y, z);
    while (*chr)
      glCallList (*chr++ + base2);  // *chr++ - 0x20 + fontlistbase
  }
}


// ge3dText

void ge3dText (const point3D* p, const char* str)
{
  ge3d_text (p->x, p->y, p->z, str);
}
#endif


// Implementation of class GEContext


GEContext::GEContext (Glyph* g, int requests)
: MonoGlyph (g)
{
  glwin_ = 0;  // created in allocate
  origcur_ = 0;
  requestoverlay_ = requests & RequestOverlay;
}


GEContext::~GEContext()
{
  delete glwin_;
}


Window* GEContext::window ()
{ 
  return glwin_;
}


void GEContext::setCursor (Cursor* csr)
{
  if (glwin_)
  { if (!origcur_)
      origcur_ = glwin_->cursor ();
    glwin_->cursor (csr);
  }
}


void GEContext::resetCursor ()
{
  if (glwin_ && origcur_)
    glwin_->cursor (origcur_);
}


void GEContext::pushCursor (Cursor* csr)
{
  if (glwin_)
  { if (!origcur_)
      origcur_ = glwin_->cursor ();
    glwin_->push_cursor ();
    glwin_->cursor (csr);
  }
}


void GEContext::popCursor ()
{
  if (glwin_)
    glwin_->pop_cursor ();
}


int GEContext::overlaySupport ()
{
  return 0;  // not yet ported; will depend on available visuals
}


void GEContext::activateOverlay (int clear)
{
  if (glwin_ && requestoverlay_)
    glwin_->activateOverlay (clear);
}


void GEContext::deactivateOverlay ()
{
  if (glwin_)
    glwin_->deactivateOverlay ();
}


void GEContext::allocate (Canvas* c, const Allocation& a, Extension&)
// told my actual allocation
{
  DEBUGNL ("GEContext::allocate");

  const Allotment& ax = a.x_allotment ();
  const Allotment& ay = a.y_allotment ();


  if (!glwin_)  // if first allocation, create a new child window for GL
  {
    glwin_ = new GLWindow (body (), c->window (), requestoverlay_);

    glwin_->canvas()->size (ax.span (), ay.span ());
    glwin_->place (ax.origin (), ay.origin ());

    glwin_->bind ();
  }
  else  // otherwise resize existing child window
  {
    glwin_->canvas ()->size (ax.span (), ay.span ());
    glwin_->place (ax.origin (), ay.origin ());

    glwin_->resize ();
  }

  DEBUGNL ("end of GEContext::allocate");

} // GEContext::allocate



// unbind
// delete glx window managed by gecontext
// when unmapping a window containing a gecontext do (in order):
// win->unmap ();  gecontext->unbind ();  win->unbind ();

void GEContext::unbind ()
{
  // not necessary with modified unbind of xwindow.c
  delete glwin_;  // is created again in next allocate
  glwin_ = 0;
  origcur_ = 0;
}



void GEContext::redraw (int)  /* hints ignored */
{
  // user request to redraw whole window
  if (glwin_)
    glwin_->rep ()->canvas_->damage_all ();
  // damage whole canvas of the glwindow
}


const char* GEContext::implementation ()
{
  return ("OpenGL");
}


int GEContext::implementationHints ()
{
#if defined(ALPHA_OSF1) || defined(ALPHA_GNU)
  return impl_requiresTrueColor;  // for 8bit displays in OpenGL
#else
  return 0;
#endif
}


int GEContext::graphicsSupport (Display* dis)
{
  if (!dis)
    dis = Session::instance ()->default_display ();

  int erb, evb;
  return glXQueryExtension (dis->rep ()->display_, &erb, &evb);
}
