/* $Id: bresenhm.c,v 1.1 1996/09/13 01:38:16 brianp Exp $ */

/*
 * Mesa 3-D graphics library
 * Version:  2.0
 * Copyright (C) 1995-1996  Brian Paul
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
 * $Log: bresenhm.c,v $
 * Revision 1.1  1996/09/13 01:38:16  brianp
 * Initial revision
 *
 */


#include "bresenhm.h"
#include "types.h"


/*
 * Evaluate Bresenham's integer line drawing algorithm.  Put each
 * coordinate generated into x[] and y[] arrays.
 *
 * Input:  x1,y1 - coordinates of first endpoint
 *         x2,y2 - coordinates of second endpoint
 * Output:  x, y - array of coordinates generated by the algorithm
 * Return:  number of values put into x[] and y[].
 */
GLuint gl_bresenham( GLcontext* ctx, GLint x1, GLint y1, GLint x2, GLint y2,
		     GLint x[], GLint y[] )
{
   register GLint dx, dy, xf, yf, a, b, c, i;

   if (x2>x1) {
      dx = x2-x1;
      xf = 1;
   }
   else {
      dx = x1-x2;
      xf = -1;
   }

   if (y2>y1) {
      dy = y2-y1;
      yf = 1;
   }
   else {
      dy = y1-y2;
      yf = -1;
   }

#define PLOT( X, Y )	x[i] = X;  y[i] = Y;

   if (dx>dy) {
      a = dy+dy;
      c = a-dx;
      b = c-dx;
      for (i=0;i<=dx;i++) {
	 PLOT( x1, y1 );
         x1 += xf;
         if (c<0) {
            c += a;
         }
         else {
            c += b;
            y1 += yf;
         }
      }
      return dx+1;
   }
   else {
      a = dx+dx;
      c = a-dy;
      b = c-dy;
      for (i=0;i<=dy;i++) {
	 PLOT( x1, y1 );
         y1 += yf;
         if (c<0) {
            c += a;
         }
         else {
            c += b;
            x1 += xf;
         }
      }
      return dy+1;
   }

#undef PLOT
}




/*
 * Evaluate Bresenham's line algorithm with stippling.
 * Input:  x1, y1, x2, y2 - endpoints of line segment
 * Output:  x, y - arrays of pixels along the line
 *          mask - indicates draw/don't draw for each pixel
 */
GLuint gl_stippled_bresenham( GLcontext* ctx, GLint x1, GLint y1, GLint x2, GLint y2,
			      GLint x[], GLint y[], GLubyte mask[] )
{
   GLint dx, dy, xf, yf, a, b, c, i;
   GLushort m;

   if (x2>x1) {
      dx = x2-x1;
      xf = 1;
   }
   else {
      dx = x1-x2;
      xf = -1;
   }

   if (y2>y1) {
      dy = y2-y1;
      yf = 1;
   }
   else {
      dy = y1-y2;
      yf = -1;
   }

#define PLOT( X, Y )							\
	m = 1 << ((ctx->StippleCounter/ctx->Line.StippleFactor) & 0xf);	\
	if (ctx->Line.StipplePattern & m) {				\
		mask[i] = 1;						\
		x[i] = X;						\
		y[i] = Y;						\
	}								\
	else {								\
		mask[i] = 0;						\
	}								\
	ctx->StippleCounter++;

   if (dx>dy) {
      a = dy+dy;
      c = a-dx;
      b = c-dx;
      for (i=0;i<=dx;i++) {
	 PLOT( x1, y1 );
         x1 += xf;
         if (c<0) {
            c += a;
         }
         else {
            c += b;
            y1 += yf;
         }
      }
      return dx+1;
   }
   else {
      a = dx+dx;
      c = a-dy;
      b = c-dy;
      for (i=0;i<=dy;i++) {
	 PLOT( x1, y1 );
         y1 += yf;
         if (c<0) {
            c += a;
         }
         else {
            c += b;
            x1 += xf;
         }
      }
      return dy+1;
   }

#undef PLOT
}
