// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  MLTextGenerate.C

  Stephane Rehel

  February 8 1998
*/

#include "tools/OString.h"

#include "FTGlyph.h"
#include "FTGlyphVectorizer.h"
#include "FTFont.h"
#include "GLTTFont.h"
#include "GLTTGlyphTriangulator.h"

#include "MLTextGenerate.h"

#include "scene/MeshRep.h"

/////////////////////////////////////////////////////////////////////////////

class MLGlyphTriangulator: public GLTTGlyphTriangulator
{
protected:
  MLTextGenerate* gen;
public:
  MLGlyphTriangulator( FTGlyphVectorizer* vec, MLTextGenerate* _gen ):
    GLTTGlyphTriangulator(vec)
    {
    gen= _gen;
    }

  virtual ~MLGlyphTriangulator()
    {}

  void triangle( FTGlyphVectorizer::POINT* p1,
                 FTGlyphVectorizer::POINT* p2,
                 FTGlyphVectorizer::POINT* p3 )
    {
    // please delegate it
    gen->triangle(p1,p2,p3);
    }
};

/////////////////////////////////////////////////////////////////////////////

MLTextGenerate::MLTextGenerate()
{
  nVertices= 0;
  nTriangles= 0;
  count_triangles= ITRUE;
  triangle_index= 0;
  polygons= 0;
}

/////////////////////////////////////////////////////////////////////////////

MLTextGenerate::~MLTextGenerate()
{}

/////////////////////////////////////////////////////////////////////////////

IBOOL MLTextGenerate::generate( MeshRep* mesh,
                                const OString& text,
                                GLTTFont& font,
                                double precision )
{
  if( mesh == 0 )
    return IFALSE;

  if( text.length() == 0 )
    return IFALSE;

  int i;

  FTGlyph** glyphs= new FTGlyph* [ text.length() ];
  FTGlyphVectorizer** vectorizers= new FTGlyphVectorizer* [ text.length() ];
  MLGlyphTriangulator** tris= new MLGlyphTriangulator* [ text.length() ];

  nVertices= 0;
  nTriangles= 0;
  count_triangles= ITRUE;
  polygons= 0;

  for( i= 0; i < text.length(); ++i )
    {
    glyphs[i]= 0;
    vectorizers[i]= 0;
    tris[i]= 0;

    int ch= (int) (unsigned char) text[i];

    FTGlyph* g= font.getFont()->getGlyph(ch);
    if( g == 0 )
      continue;

    glyphs[i]= 0;
    vectorizers[i]= new FTGlyphVectorizer;

    FTGlyphVectorizer*& v= vectorizers[i];

    v->setPrecision(precision);

    if( ! v->init(g) )
      {
      delete v;
      v= 0;
      continue;
      }

    if( ! v->vectorize() )
      {
      delete v;
      v= 0;
      continue;
      }

    tris[i]= new MLGlyphTriangulator(v,this);

    if( ! tris[i]->triangulate() )
      {
      delete tris[i];
      tris[i]= 0;
      continue;
      }

    for( int c= 0; c < v->getNContours(); ++c )
      {
      FTGlyphVectorizer::Contour* contour= v->getContour(c);
      if( contour == 0 )
        continue;
      for( int j= 0; j < contour->nPoints; ++j )
        {
        FTGlyphVectorizer::POINT* point= contour->points + j;
        point->data= (void*) new int;
        ++nVertices;
        *((int*) point->data)= nVertices;
        }
      }
    }

  mesh->alloc( nVertices, 0, nTriangles );

  count_triangles= IFALSE;
  polygons= &(mesh->polygons);
  triangle_index= 1;

  double x= 0.;
  for( i= 0; i < text.length(); ++i )
    {
    GLTTGlyphTriangulator* t= tris[i];
    FTGlyphVectorizer* v= vectorizers[i];

    if( t == 0 || v == 0 )
      continue;

    for( int c= 0; c < v->getNContours(); ++c )
      {
      FTGlyphVectorizer::Contour* contour= v->getContour(c);
      if( contour == 0 )
        continue;
      for( int j= 0; j < contour->nPoints; ++j )
        {
        FTGlyphVectorizer::POINT* point= contour->points + j;
        Point p( x + point->x, point->y, 0. );

        int vi= *((int*) point->data);

        mesh->vertices[ vi ].point= p;
        }
      }

    tris[i]->triangulate();

    x+= v->getAdvance();
    }

  count_triangles= ITRUE;
  polygons= 0;

  for( i= 0; i < text.length(); ++i )
    {
    delete tris[i];
    tris[i]= 0;

    FTGlyphVectorizer* v= vectorizers[i];
    if( v != 0 )
      {
      for( int c= 0; c < v->getNContours(); ++c )
        {
        FTGlyphVectorizer::Contour* contour= v->getContour(c);
        if( contour == 0 )
          continue;
        for( int j= 0; j < contour->nPoints; ++j )
          {
          FTGlyphVectorizer::POINT* point= contour->points + j;
          delete point->data;
          point->data= 0;
          }
        }

      delete vectorizers[i];
      vectorizers[i]= 0;
      }

    delete glyphs[i];
    glyphs[i]= 0;
    }

  delete tris;
  tris= 0;

  delete vectorizers;
  vectorizers= 0;

  delete glyphs;
  glyphs= 0;

  return ITRUE;
}

/////////////////////////////////////////////////////////////////////////////

void MLTextGenerate::triangle( FTGlyphVectorizer::POINT* p1,
                               FTGlyphVectorizer::POINT* p2,
                               FTGlyphVectorizer::POINT* p3 )
{
  if( count_triangles )
    {
    ++nTriangles;
    return;
    }

  if( polygons == 0 )
    return;
  if( triangle_index > polygons->getSize() || p1 == 0 || p2 == 0 || p3 == 0 )
    return;

  MLPolygon& polygon= (*polygons)[triangle_index];

  int v1= *((int*)(p1->data));
  int v2= *((int*)(p2->data));
  int v3= *((int*)(p3->data));

  if( v1 < 0 || v1 > nVertices ||
      v2 < 0 || v2 > nVertices ||
      v3 < 0 || v3 > nVertices )
    return;

  polygon.allocTriangle( v1, v2, v3 );

  ++triangle_index;
}


/////////////////////////////////////////////////////////////////////////////
