// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  MLSaveGeometry.C

  Stephane Rehel
  April 22 1997
*/

#include <string.h>
#include <errno.h>

#include "tools/Filename.h"
#include "tools/ListFiles.h"

#include "kw/KWDialog.h"

#include "interface/MLIOProgressDialog.h"

#include "model/ModelModule.h"

#include "scene/file/MLDatabasesList.h"
#include "scene/file/MLDatabase.h"
#include "scene/file/MLAbstractMeshIO.h"

#include "MLSaveGeometry.h"
#include "SaveGeometryDlg.h"
#include "WritingProgressDlg.h"

/////////////////////////////////////////////////////////////////////////////

static const char* errorString= "*ERROR*";

/////////////////////////////////////////////////////////////////////////////

MLSaveGeometry::MLSaveGeometry( ModelModule* _model ):
  MLGlobalDialog(_model)
{
  model= _model;
  dblist= 0;
  files= new ListFiles;
}

/////////////////////////////////////////////////////////////////////////////

MLSaveGeometry::~MLSaveGeometry()
{
  delete dblist;
  dblist= 0;

  delete files;
  files= 0;
}

/////////////////////////////////////////////////////////////////////////////

// return the abstract object read
// return 0 if cancelled or error
IBOOL MLSaveGeometry::run( MLAbstractMesh* ao )
{
  if( ao == 0 )
    return IFALSE;

  delete dblist;
  dblist= new MLDatabasesList;
  dblist->build();

  if( dblist->get().empty() )
    {
    KWError( "Error",
               OString("No database available!\nCheck: ")
             + MLDatabasesList::getDatabasesResourceFilename() );
    return IFALSE;
    }

  if( ! MLGlobalDialog::create(SaveGeometryDlg::data) )
    return IFALSE;

  SIListIterator<MLDatabase> li(dblist->get());
  while( ! li.eol() )
    {
    MLDatabase* db= li++;
    kwDialog->appendListText( SaveGeometryDlg::databases,
                              db->getBasePath() );
    }
  kwDialog->setInteger( SaveGeometryDlg::databases,
                        dblist->getCurrentDatabasePosition());

  OString dbpath= kwDialog->getText(SaveGeometryDlg::databases);
  currentDatabase= dblist->getDatabase(dbpath);

  updateFilesList();

  kwDialog->setText( SaveGeometryDlg::filename,
                     MLAbstractMeshIO::getFilename(ao) );

  kwDialog->focus(SaveGeometryDlg::filename);

  int outcome= kwDialog->run();

  OString filename= kwDialog->getText(SaveGeometryDlg::filename);
  if( filename == errorString )
    filename= "";

  if( outcome == KWDialog::CANCEL || filename.length() == 0 )
    {
    MLGlobalDialog::close();
    return IFALSE;
    }

  IBOOL ok= save(filename,ao);

  MLGlobalDialog::close();

  return ok;
}

/////////////////////////////////////////////////////////////////////////////

IBOOL MLSaveGeometry::save( const OString& _filename,
                            MLAbstractMesh* ao )
{
  if( currentDatabase == 0 || _filename.length() == 0 )
    return IFALSE;

  OString filename= _filename;
  if( Filename(filename).getExtension().length() == 0 )
    filename += OString(MLDatabase::geometryExtension);

  model->stopLightingProcess();

  MLIOProgressDialog* progress= new MLIOProgressDialog(model);
  progress->create( WritingProgressDlg::data,
                    WritingProgressDlg::progress,
                    WritingProgressDlg::label );
  progress->flush();

  MLAbstractMeshIO aoWriter(currentDatabase,model->getScene(),progress);

  model->pushWaitCursor();

  IBOOL outcome= aoWriter.write(ao,filename);

  delete progress;
  progress= 0;

  model->popWaitCursor();

  if( ! outcome )
    {
    KWError( "Error",
               OString("Unable to save:\n")
             + filename + OString("\n")
             + OString(strerror(errno)) );
    return IFALSE;
    }

  return ITRUE;
}

/////////////////////////////////////////////////////////////////////////////

void MLSaveGeometry::updateFilesList()
{
  if( kwDialog == 0 )
    return;

  kwDialog->clearList(SaveGeometryDlg::list);

  MLDatabase* database= currentDatabase;
  if( database == 0 )
    return;

  MLDatabasesList::setCurrent( currentDatabase->getBasePath() );

  OString path= database->getGeomPath();
  OString pattern= OString("*") + OString(MLDatabase::geometryExtension);

  if( files->list(path,pattern,IFALSE) != ListFiles::SUCCESS )
    {
    kwDialog->appendListText( SaveGeometryDlg::list, errorString );
    return;
    }

  IListIterator<ListFiles::FileItem> li(files->getFiles());
  while( ! li.eol() )
    {
    ListFiles::FileItem* item= li.next();
    kwDialog->appendListText( SaveGeometryDlg::list, item->name );
    }
}

/////////////////////////////////////////////////////////////////////////////

IBOOL MLSaveGeometry::kwCallback( KWEvent& event )
{
  if( event.type == KWEvent::SELECT &&
      event.widget == SaveGeometryDlg::list )
    {
    OString filename= kwDialog->getText(SaveGeometryDlg::list);
    if( filename == OString(errorString) )
      filename= "";

    kwDialog->setText( SaveGeometryDlg::filename, filename );
    kwDialog->focus(SaveGeometryDlg::filename);
    return ITRUE;
    }

  if( event.type == KWEvent::SELECT &&
      event.widget == SaveGeometryDlg::databases )
    {
    OString dbpath= kwDialog->getText(SaveGeometryDlg::databases);
    currentDatabase= dblist->getDatabase(dbpath);
    updateFilesList();
    return ITRUE;
    }

  return MLGlobalDialog::kwCallback(event);
}

/////////////////////////////////////////////////////////////////////////////

