// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  KWFileDialogManager.C

  Stephane Rehel

  August 11 1997
*/

#include "tools/Path.h"
#include "tools/ListFiles.h"

#include "graphics/KeyCodes.h"

#include "KWFileDialogManager.h"
#include "KWDialog.h"

static const IBOOL hide_dot_files= ITRUE;

/////////////////////////////////////////////////////////////////////////////

KWFileDialogManager::KWFileDialogManager()
{
  dialog= 0;
  directories= files= location= selection= ok= cancel= 0;
  size= date= user= 0;

  path= new Path;
  listfiles= new ListFiles;

  filename= "";
  pattern= "";
}

/////////////////////////////////////////////////////////////////////////////

KWFileDialogManager::~KWFileDialogManager()
{
  dialog= 0;
  directories= files= location= selection= ok= cancel= 0;

  delete listfiles; listfiles= 0;

  delete path; path= 0;
  pattern= "";
}

/////////////////////////////////////////////////////////////////////////////

void KWFileDialogManager::init( KWDialog* _dialog,
                                unsigned int _directories,
                                unsigned int _files,
                                unsigned int _location,
                                unsigned int _selection,
                                unsigned int _ok,
                                unsigned int _cancel,
                                unsigned int _size /*= 0*/,
                                unsigned int _date /*= 0*/,
                                unsigned int _user /*= 0*/ )
{
  dialog= _dialog;
  directories= _directories;
  files= _files;
  location= _location;
  selection= _selection;
  ok= _ok;
  cancel= _cancel;
  size= _size;
  date= _date;
  user= _user;

  *path= "./";
  pattern= "*";
}

/////////////////////////////////////////////////////////////////////////////

// return ITRUE if user has entered a real filename, and return
// the effective full filename

IBOOL KWFileDialogManager::validate()
{
  filename= "";

  if( dialog == 0 )
    return IFALSE;

  OString fn= dialog->getText(selection);
  if( fn.length() == 0 )
    return IFALSE;

  fn= ListFiles::expandFilename(fn);

  if( ListFiles::isAccessibleDirectory(fn) )
    {
    if( ! ListFiles::changeToDirectory(fn) )
      return IFALSE;
    *path= ListFiles::getCurrentDirectory();
    OString pp= path->get();
    listfiles->list(pp,pattern);
    _updateDirectories();
    dialog->setText( selection, pattern );
    dialog->focus(selection);
    return IFALSE;
    }

  OString p= "",
          f= "";
  ListFiles::splitFilename( fn, p, f );
  if( f.length() == 0 )
    return IFALSE;

  if( ! ListFiles::changeToDirectory(p) )
    return IFALSE;

  if( f.locateChar('*') == -1 && f.locateChar('?') == -1 )
    {
    // user has entered an existing file as
    // '/usr/local/include/toto.h' or 'toto.h'
    *path= p;
    dialog->setText( selection, f );
    dialog->focus(selection);
    filename= path->get() + f;
    return ITRUE;
    }

  if( listfiles->list(p,f) != ListFiles::SUCCESS )
    return IFALSE;

  // we assume this is a glob expression
  *path= p;
  pattern= f;

  dialog->setText( selection, f );
  dialog->focus(selection);

  _updateDirectories();

  return IFALSE;
}

/////////////////////////////////////////////////////////////////////////////

void KWFileDialogManager::updateDirectories()
{
  OString pp= path->get();
  listfiles->list( pp, pattern );
  _updateDirectories();
}

/////////////////////////////////////////////////////////////////////////////

void KWFileDialogManager::_updateDirectories()
{
  if( dialog == 0 )
    return;

  dialog->setText( location, path->get() );

  dialog->clearList(directories);
  dialog->clearList(files);

  if( ! ListFiles::changeToDirectory(path->get()) )
    return;

  IListIterator<ListFiles::FileItem> li(listfiles->getFiles());
  while( ! li.eol() )
    {
    ListFiles::FileItem* fi= li++;
    if( hide_dot_files )
      {
      if( fi->name[0] == '.' && fi->name != ".." )
        continue;
      }

    if( fi->isDirectory )
      {
      if( !(fi->name == ".." && path->get() == "/") )
        dialog->appendListText( directories, fi->name );
      }
     else
      dialog->appendListText( files, fi->name );
    }
}

/////////////////////////////////////////////////////////////////////////////

void KWFileDialogManager::setPattern( const OString& _pattern )
{
  pattern= _pattern;

  if( dialog != 0 )
    dialog->setText( selection, pattern );
}

/////////////////////////////////////////////////////////////////////////////

void KWFileDialogManager::setPath( const OString& _path )
{
  *path= ListFiles::expandFilename(_path);
}

/////////////////////////////////////////////////////////////////////////////

static ListFiles::FileItem* findListItem( IList<ListFiles::FileItem>& list,
                                          const OString& filename )
{
  IListIterator<ListFiles::FileItem> li(list);
  while( ! li.eol() )
    {
    ListFiles::FileItem* item= li.next();
    if( item->name == filename )
      return item;
    }

  return 0;
}

/////////////////////////////////////////////////////////////////////////////

void KWFileDialogManager::updateSizeDateUser()
{
  if( dialog == 0 )
    return;

  OString path= dialog->getText(location);
  OString file= dialog->getText(selection);
  OString filename= path + file;

  ListFiles::FileItem* item= findListItem(listfiles->getFiles(),file);

  if( item == 0 )
    {
    if( size != 0 ) dialog->setText(size,"");
    if( date != 0 ) dialog->setText(date,"");
    if( user != 0 ) dialog->setText(user,"");
    return;
    }

  if( size != 0 )
    dialog->setText( size, OString::itoa(int(item->st_size)) );
  if( date != 0 )
    dialog->setText( date, item->getDateString() );
  if( user != 0 )
    dialog->setText( user, item->getUser() );
}

/////////////////////////////////////////////////////////////////////////////

IBOOL KWFileDialogManager::handleKWEvent( KWEvent& event )
{
  if( files != 0 &&
      event.type == KWEvent::SELECT &&
      event.widget == files )
    {
    OString f= dialog->getText(files);
    dialog->setText( selection, f );
    updateSizeDateUser();
    return ITRUE;
    }

  if( files != 0 &&
      event.type == KWEvent::DOUBLE_SELECT &&
      event.widget == files )
    {
    OString f= dialog->getText(files);
    dialog->setText( selection, f );

    if( validate() )
      dialog->quit(KWDialog::VALIDATE);

    return ITRUE;
    }

  if( directories != 0 &&
      event.type == KWEvent::SELECT &&
      event.widget == directories )
    {
    OString f= dialog->getText(directories);

    dialog->setText( selection, f + OString("/") );
    updateSizeDateUser();
    return ITRUE;
    }

  if( directories != 0 &&
      event.type == KWEvent::DOUBLE_SELECT &&
      event.widget == directories )
    {
    OString f= dialog->getText(directories);

    dialog->setText( selection, f + OString("/") );

    validate();

    updateSizeDateUser();

    return ITRUE;
    }

  if( (ok != 0 &&
       event.type == KWEvent::SELECT &&
       event.widget == ok) ||
      (event.type == KWEvent::KEY &&
       event.key == KeyCodes::ENTER) )
    {
    if( validate() )
      {
      dialog->quit(KWDialog::VALIDATE);
      }

    return ITRUE;
    }

  if( (cancel != 0 &&
       event.type == KWEvent::SELECT &&
       event.widget == cancel) ||
      (event.type == KWEvent::KEY &&
       event.key == KeyCodes::ESC) )
    {
    dialog->quit(KWDialog::CANCEL);
    return ITRUE;
    }

  return IFALSE;
}

/////////////////////////////////////////////////////////////////////////////

