// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  Canvas.h

  Stephane Rehel
  July 30 1996
*/

#ifndef __Canvas_h
#define __Canvas_h

#include "tools/SIList.h"
#include "tools/IRectangle.h"
#include "tools/OString.h"
#include "graphics/MLEventHandler.h"
#include "graphics/MLKeyboardHandler.h"
#include "kw/KWCallbackClass.h"

class VDialog;
class VWindow;
class MLWindow;
class MLEvent;
class VWidget;
class GWidget;
class SystemWindow;
class VCartouche;
class KWDialog;
class VCard;

class Xinfo;

class KBModule;
class KBCanvas;

/////////////////////////////////////////////////////////////////////////////

class Canvas: public MLEventHandler, public MLKeyboardHandler,
              public KWCallbackClass
{
  friend class GWidget;

public:
  static const int xorRectangleSize;
  static const int NOT_INTO;
  static const int INTO;
  static const int UP;
  static const int DOWN;
  static const int RIGHT;
  static const int LEFT;

  static OString canvasExtension;

protected:
  Xinfo* xinfo;
  SystemWindow* root;
  SystemWindow* systemWindow;
  VCartouche* cartouche;
  KWDialog* popup;

  VDialog* dialog;
  VCard* activeCard;

  SIList<GWidget> widgets;
  SIList<GWidget> selectedWidgets;
  SIList<GWidget> clipboard;

  typedef SIList<GWidget> GGroup;
  SIList<GGroup> groups;

  IPoint dialogPosition;
  IVector dialogSize;
  IBOOL frame;
  IBOOL killButton;
  OString title;
  OString filename, name;
  OString C_filename, h_filename;
  IBOOL has_C_filename, has_h_filename;
  IBOOL write_C_file;
  IBOOL validateCancel;
  int validateID, cancelID;
  int granularity, defaultGranularity;
  IBOOL drawGroups;

  IRectangle cardsRectangle;

  int mousePos;
  IVector drag_pos;
  IVector tmp_drag_pos;
  IPoint drag_origin;
  IPoint drag_origin2;
  int drag_pos_gravity;
  int dragging;

  IBOOL modified, h_modified;
  int top_margin, bottom_margin, right_margin, left_margin;

  KBCanvas* kbCanvas;
  KBModule* kbModule;

public:
  Canvas( SystemWindow* _root, KBCanvas* _kbCanvas );
  virtual ~Canvas();

  void destroy();
  void recreate();
  void create();

  void newDialog();

  // in CanvasDraw.C
  void drawInfos();
  void draw_now();
  void draw();
  void drawXORRectangle( const IRectangle& _rect,
                         IBOOL correct = ITRUE  );
  void xorRectangle( GWidget* gw, const IRectangle& rect );
  void xorSelected( GWidget* gw );
  void xorSelected();

  IBOOL isRadioBoxGroup( GGroup* g );
  GGroup* findGroup( GWidget* gw );
  void removeFromGroup( GWidget* gw );
  void selectGroup( GGroup* g, IBOOL toggle );
  GGroup* createGroup();

  GWidget* findWidget( int id );

  SIList<GWidget>& getWidgets()
    {
    return widgets;
    }

  // in CanvasMouse.C
  GWidget* getMousePosition( const IPoint& pos, int& mousePos );
  void mousePress( MLEvent& event );
  void mouseMotion( MLEvent& event );
  void mouseRelease( MLEvent& event );

  void moveWidgets( const IVector& displ );

  IBOOL eventHandler( MLWindow*, MLEvent& );
  IBOOL handleMLKeyboard( MLEvent& event );

  const OString& getFilename() const
    {
    return filename;
    }

  void writeCanvasAs();
  void writeCanvas();
  IBOOL writeCanvas( const OString& _filename );

  void readCanvas();
  IBOOL readCanvas( const OString& _filename );

private:
  void _encodeCanvas();

public:
  int findEncodingLength();
  IBOOL encodeCanvas_C();
  IBOOL encodeCanvas_h();

  IBOOL hasCards();
  IRectangle getBounding();
  void changeDialogSize( const IVector& newSize );

  void makeLinks();

  virtual IBOOL kwCallback( KWEvent& e );

  static OString getCardName( VCard* card );

  IBOOL editButtonCallback( KWEvent& e );
  IBOOL editButton( GWidget* gw );
  IBOOL editLabel( GWidget* gw );
  IBOOL editCheckBox( GWidget* gw );
  IBOOL editGroup( GWidget* gw );

  IBOOL editEditCallback( KWEvent& e );
  IBOOL editEdit( GWidget* gw );

  IBOOL editDialogCallback( KWEvent& e );
  IBOOL editDialog();

  IBOOL editLine( GWidget* gw );
  IBOOL editTextBox( GWidget* gw );

  IBOOL editListCallback( KWEvent& e );
  IBOOL editList( GWidget* gw );

  IBOOL editFolderListCallback( KWEvent& e );
  IBOOL editFolderList( GWidget* gw );

  IBOOL editArea( GWidget* gw );
  IBOOL editProgress( GWidget* gw );
  IBOOL editBar( GWidget* gw );
  IBOOL editRadioButton( GWidget* gw );

  IBOOL editAlignCallback( KWEvent& e );
  IBOOL editAlign();

  IBOOL editValueEditCallback( KWEvent& e );
  IBOOL editValueEdit( GWidget* gw );

//  void createPopup();
//  IBOOL popupCallback( KWEvent& e );
  int getGranularity() const
    {
    return granularity;
    }
  int getDefaultGranularity() const
    {
    return defaultGranularity;
    }
  void setGranularity( int g );
  void setActiveCard( VCard* card );
  VCard* getActiveCard() const
    {
    return activeCard;
    }

  void sortCards();

  IBOOL editPrefs();

  IBOOL editPicture( GWidget* gw );

  IBOOL editChooserCallback( KWEvent& e );
  IBOOL editChooser( GWidget* gw );

  IBOOL editRangeBar( GWidget* gw );

  IBOOL editValueBarCallback( KWEvent& e );
  IBOOL editValueBar( GWidget* gw );

// actions
  void deleteWidgets();
  void deleteCard();
  void end_newWidget( GWidget* w );
  void newCard( IBOOL insertIt = IFALSE );
  void newPicture();
  void newButton();
  void newChooser();
  void newLabel();
  void newCheckBox();
  void newEdit();
  void newValueEdit();
  void newRadioButton();
  void newVertBar();
  void newHorizBar();
  void newTextBox();
  void newList();
  void newFolderList();
  void newHorizLine();
  void newVertLine();
  void newGroup();
  void newArea();
  void newProgress();
  void newRangeBar();
  void newValueBar();
  void expandCanvas();
  void shrinkCanvas();
  void duplicateWidgets();
  void availableWidgets();
  void editWidget();
  void editDialogAction();
  void centerAllWidgets();
  void raiseWidget( IBOOL yes, GWidget* gw );
  void raiseWidgets( IBOOL yes );
  void groupWidgets();
  void ungroupWidgets();
  void alignWidgets( int horiz, int vert );
  IBOOL editCard();

  void copy_to_clipboard();
  void copy_from_clipboard();
};

/////////////////////////////////////////////////////////////////////////////

#endif // ifndef __Canvas_h
