#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <signal.h>
#include <errno.h>
#include <termios.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>

#include <config.h>
#include <support.h>
#include <xcio.h>

#include <sysmsg.h>
#include <env.h>
#include <log.h>
#include <chat.h>
#include <option.h>
#include "device.h"

static pid_t dialerPid;
static size_t replyMaxLen;

struct dialopt_s {
    struct numlist_s {
	char *number;
	int retry;
	struct numlist_s *next;
    } *numhp;
    char *type;
    u_int32_t interval;
    u_int32_t rate, timeout;
};

static struct numlist_s *numberCp;
static int dialRetry = -1;
static struct dialopt_s dialOpt;

static struct modeminfo_s {
    struct modeminfo_s *next;
    char *name;
    enum {MI_STR, MI_FLAG, MI_NUM, MI_REPLY} type;
    int len;
    union {
	char *str;
	u_int32_t flags;
	u_int32_t num;
    } d;
} *miHead;

static struct {
    const char *name;
    char *str;
} flagList[]={
    {"Ok", NULL},
    {"Connect", NULL},
    {"Ring", NULL},
    {"Error", NULL},
    {"Busy", NULL},
    {"NoCarrier", NULL},
    {"NoDialTone", NULL},
    {"NoAnswer", NULL},
    {"Delayed", NULL},
    {"Timeout", NULL}
};

enum {
    RE_OK, RE_CONNECT, RE_RING, RE_ERROR, RE_BUSY,
    RE_NOCARRIER, RE_NODIALTONE, RE_NOANSWER, RE_DELAYED, RE_MAX
};

#if 0
void
DumpModemInfo()
{
    struct modeminfo_s *mip=miHead;
    int n;
    
    while (mip) {
	printf("<%s>\t\t", mip->name);
	switch (mip->type) {
	case MI_REPLY:
	case MI_STR:
	    if (mip->d.str) printf("\"%s\"", mip->d.str);
	    for (n = 0; n < RE_MAX; n ++) {
		if (mip->d.str == flagList[n].str) {
		    printf("(%d)", n);
		    break;
		}
	    }
	    break;
	case MI_NUM:
	    printf("%d", mip->d.num);
	    break;
	case MI_FLAG:
	    printf("%x", mip->d.flags);
	    break;
	}
	printf("\n");
	mip = mip->next;
    }
}
#endif

static bool_t
EnvNumber(int argc, char **argv, char *outs)
{
    if (argc) return FALSE;
    if (numberCp == NULL) numberCp = dialOpt.numhp;
    if (numberCp) {
	outs += SprintF(outs, "%s", numberCp->number);
	if (numberCp->retry > 0)
	    SprintF(outs, "/%d",
		    dialRetry < 0 ? numberCp->retry: dialRetry);
    }
    return TRUE;
}

static bool_t
EnvNumList(int argc, char **argv, char *outs)
{
    struct numlist_s *nsp, *nst;
    int a;
    char *p;

    nsp = dialOpt.numhp;
    if (!argc) {
	p = outs;
	while (nsp) {
	    p += SprintF(p, "%s", nsp->number);
	    if (nsp->retry > 0)
		p += SprintF(p, "/%d", nsp->retry);
	    nsp = nsp->next;
	    if (nsp) p += SprintF(p, " ");
	}
	return FALSE;
    }
    while (nsp) {
	Free(nsp->number);
	nst = nsp->next;
	Free(nsp);
	nsp = nst;
    }
    nst = NULL;
    for (a = 1; a < argc; a ++) {
	nsp = TALLOC(struct numlist_s);
	nsp->next = NULL;
	nsp->number = Strdup(argv[a]);
	if ((p = strrchr(nsp->number, '/')) != NULL) {
	    *p = '\0';
	    p ++;
	    nsp->retry = atoi(p);
	} else nsp->retry = 1;
	if (!nst) dialOpt.numhp = nsp;
	else nst->next = nsp;
	nst = nsp;
    }
    numberCp = NULL;
    return TRUE;
}

static int
ModemInfoProc(char *line, int secure)
{
    struct modeminfo_s *mip;
    char *p, *data;

    if ((p = strpbrk(line, "\r\n")) != NULL) *p = '\0';
    p = line;
    while (*p && !strchr("\t ", *p)) p ++;
    if (!*p) return(-1);
    *p = '\0';
    p ++;
    while (*p && strchr("\t ", *p)) p ++;
    if (!*p) return(-1);
    data = p;
    if (!strcasecmp("include", line)) {
	int secure=0;
	FILE *fp;

	fp = OpenFile(data, "modem", &secure);
	return(fp ? LoadFile(fp, ModemInfoProc, secure, NULL): -1);
    }
    
    mip = miHead;
    while (mip) {
	if (!strncasecmp(mip->name, line, strlen(mip->name))) break;
	mip = mip->next;
    }
    if (!mip) {
	mip = TALLOC(struct modeminfo_s);
	mip->next = miHead;
	miHead = mip;
	mip->name = Strdup(line);
	mip->d.str = NULL;
    }
    if ((p = strchr(data, '"')) != NULL) {
	int n;

	mip->type = MI_STR;
	data = p + 1;
	if ((p = strchr(data, '"')) != NULL) {
	    *p = '\0';
	    if (mip->d.str) free(mip->d.str);
	    mip->d.str = Strdup(data);
	}
	for (n = 0; n < RE_MAX; n ++) {
	    if (!strcasecmp(flagList[n].name, mip->name)) {
		size_t len;

		flagList[n].str = mip->d.str;
		len = strlen(mip->d.str);
		if (replyMaxLen < len) replyMaxLen = len;
		break;
	    }
	}
    } else if (isdigit(*data) && !strchr(data, ',')) {
	mip->type = MI_NUM;
	mip->d.num = atoi(data);
    } else {
	int n;

	mip->type = MI_FLAG;
	p = strtok(data, ", ");
	while (p) {
	    for (n = 0; n < RE_MAX; n ++) {
		if (!strcmp(p, flagList[n].name)) {
		    mip->d.flags |= 1 << n;
		    break;
		}
	    }
	    p = strtok(NULL, ", ");
	}
    }
    return(0);
}

int
LoadModemInfo(char *name)
{
    struct modeminfo_s *mip, *mip1;
    int secure=0;
    FILE *fp;

    mip = miHead;
    while (mip) {
	mip1 = mip->next;
	if (mip->type != MI_NUM && mip->d.str) free(mip->d.str);
	free(mip);
	mip = mip1;
    }
    miHead = NULL;
    replyMaxLen = 0;

    fp = OpenFile(name, "modem", &secure);
    return(fp ? LoadFile(fp, ModemInfoProc, secure, NULL): -1);
}

char *
ModemGetStr(const char *name)
{
    struct modeminfo_s *mip=miHead;

    while (mip) {
	if (mip->type == MI_STR
	    && !strcasecmp(name, mip->name))
	    return(mip->d.str);
	mip = mip->next;
    }
    return(NULL);
}

int
ModemGetNum(const char *name)
{
    struct modeminfo_s *mip=miHead;

    while (mip) {
	if (mip->type == MI_NUM
	    && !strcasecmp(name, mip->name))
	    return(mip->d.num);
	mip = mip->next;
    }
    return(-1);
}

char *
ModemGetName()
{
    return(ModemGetStr("Name"));
}

void
ModemLog(const char *title, char *s, int n)
{
    char *p=s;

    Logf(LOG_CHAT, "%s \"", title);
    while (*p && (p - s < n)) {
	if (strchr("\r\n", *p))
	    Logf(LOG_CHAT, "\\%c", *p == '\r' ? 'r': 'n');
	else
	    Logf(LOG_CHAT, "%c", *p);
	p ++;
    }
    Logf(LOG_CHAT, "\"\n");
}

static int
ModemWrite(int fd, char *s, int n)
{
    int i;

    if (ISLOG(LOG_CHAT)) ModemLog("write", s, n);
    for (i = 0; i < n; i ++, s ++) {
	write(fd, s, 1);
	if (dialOpt.rate) usleep(dialOpt.rate * 1000);
    }
}

static int
DialerCS(int fd, char cmd, int n)
{
    int i;

    switch (cmd) {
    case 'd':	/* n*100ms delay */
	if (ISLOG(LOG_CHAT)) Logf(LOG_CHAT, "delay %dms\n", n * 100);
	usleep(n * 100000);
	break;
    case 'w':
	if (ISLOG(LOG_CHAT)) Logf(LOG_CHAT, "wait %dms\n", n * 100);
	DialerRecv(fd, NULL, n * 100, ~0);
	break;
    default:
	return(-1);
    }
    return(0);
}

int
DialerRecv(int fd, char *istr, time_t msec, u_int32_t rcbits)
{
    static size_t rSiz;
    static char *rBuf;
    int n;
    size_t cur=0, len=0;
    char mc=0;
    fd_set rfds;
    struct timeval tv;

    if (istr) {
	len = strlen(istr);
	if (ISLOG(LOG_CHAT)) Logf(LOG_CHAT, "ignore until '%s'\n", istr);
    }
    if ((cur = replyMaxLen) < len) cur = len;
    cur += 10;
    if (rSiz < cur) {
	rSiz = cur;
	rBuf = Realloc(rBuf, rSiz + 1);
    }
    memset(rBuf, 0, rSiz + 1);

    /* (modem) -- [read] --> rBuf */
    tv.tv_sec = msec / 1000;
    tv.tv_usec = (msec % 1000) * 1000;
    cur = 0;
    while (1) {
	FD_ZERO(&rfds);
	FD_SET(fd, &rfds);
	n = select(fd + 1, &rfds, NULL, NULL, msec ? &tv: NULL);
	if (n < 0 && errno == EINTR) continue;
	if (n == 0) {
	    if (ISLOG(LOG_CHAT))
		Logf(LOG_CHAT, "dialer: timeout(%dms)\n", msec);
	    return(RE_MAX);	/* timeout */
	}
	if (read(fd, &mc, 1) <= 0) {
	    if (errno == EINTR) continue;
	    return(-1);
	}
	if (cur < rSiz) rBuf[cur ++] = mc;
	else {
	    memcpy(rBuf, rBuf + 1, rSiz - 1);
	    rBuf[cur - 1] = mc;
	}
	rBuf[cur] = '\0';
	if (strchr("\r\n", mc)) cur = 0;
	if (istr) {
	    if (strstr(rBuf, istr)) istr = NULL;
	} else {
	    for (n = 0; n < RE_MAX; n ++) if ((1 << n) & rcbits) {
		if (strstr(rBuf, flagList[n].str)) {
		    if (ISLOG(LOG_CHAT))
			ModemLog("found", rBuf, strlen(rBuf));
		    return(n);
		}
	    }
	}
	if (!cur && ISLOG(LOG_CHAT))
	    ModemLog("read ", rBuf, strlen(rBuf));
    }
    return(-1);
}

char *
DialerSend(int fd, const char *name)
{
    static char *sBuf, *cBuf;
    static size_t sSiz, cSiz;
    size_t len;
    int num;
    char *cp, *sp;

    if ((cp = ModemGetStr(name)) == NULL) {
	ConsoleMsg(MS_E_MODEM_UNKNOWN, name);
	return(NULL);
    }
    len = strlen(cp) + 2;	/* 2: sizeof("\r") */
    if (len > cSiz) {
	cSiz = len;
	cBuf = Realloc(cBuf, cSiz);
    }
    strcpy(cBuf, cp);
    strcat(cBuf, "\r");
    cp = cBuf;

    len = cSiz + 16;	/* 16: sizeof("1234-567-8901") + alpha */
    if (len > sSiz) {
	sSiz = len;
	sBuf = Realloc(sBuf, sSiz + 1);
    }
    sp = sBuf;
    /* cBuf -- [extract] --> sBuf -- [write] --> (modem) */
    while (*cp) {
	switch(*cp) {
	case '\\':
	    num = 0;
	    cp ++;
	    while (isdigit(*cp) && *cp) {
		num = (num * 10) + (*cp - '0');
		cp ++;
	    }
	    switch (*cp) {
	    case 0:
		return(NULL);
	    case 'r':
		*cp = '\r';
		goto add_char;
	    case '\\':
		goto add_char;
	    case 'p':
		strcpy(sp, numberCp->number);
		sp += strlen(numberCp->number);
		break;
	    default:
		if (sp - sBuf > 0) {
		    ModemWrite(fd, sBuf, sp - sBuf);
		    *sp = '\0';
		    sp = sBuf;
		}
		DialerCS(fd, *cp, num ? num: 1);
		break;
	    }
	    break;
	default:
	add_char:
	    *sp ++ = *cp;
	}
	cp ++;
    }
    if (sp - sBuf > 0) {
	ModemWrite(fd, sBuf, sp - sBuf);
	*sp = '\0';
    }
    if ((sp = strpbrk(sBuf, "\r\n")) != NULL) *sp = '\0';
    return(sBuf);
}

static int
DialerInit(int fd)
{
    int n, ret=RE_ERROR, ito;
    char *rs, *ie;
    u_int32_t rcbits;

    rcbits = ~((1<<RE_CONNECT)|(1<<RE_BUSY)|(1<<RE_DELAYED));
    ie = ModemGetStr("InitialEcho");
    ito = ModemGetNum("InitialTimeout");
    if (ie && !*ie) ie = NULL;
    for (n = 0; n < 3; n ++) {
	if ((rs = DialerSend(fd, "Initialize")) == NULL) break;
	if (!ie || strcmp(ie, "\\!")) rs = ie;
	ret = DialerRecv(fd, rs, ito, rcbits);
	if (ret == RE_OK) break;
/*	tcflush(fd, TCIOFLUSH);*/
	usleep(1000000);
    }
    if (ret != RE_OK) ConsoleMsg(MS_E_MODEM_INIT,
				 ret < 0 ? "Unknown": flagList[ret].name);
    return(ret);
}

static int
DialerCall(int fd)
{
    int ret;
    char dial[100];
    u_int32_t rcbits;

    ret = DialerInit(fd);
    if (ret != RE_OK) return(CHAT_ABORT);

    if (dialOpt.type && *dialOpt.type)
      snprintf(dial, sizeof(dial), "%.30sDial", dialOpt.type);
    else strcpy(dial, "Dial");
    DialerSend(fd, dial);
    rcbits = ~((1<<RE_OK));
    ret = DialerRecv(fd, NULL,
		     dialOpt.timeout > 0 ? dialOpt.timeout: 60000,
		     rcbits);
    switch(ret) {
    case RE_CONNECT:
	return(CHAT_DONE);
    case RE_BUSY:
    case RE_DELAYED:
	ConsoleMsg(MS_E_MODEM_REDIAL,
		   ret < 0 ? "Unknown": flagList[ret].name);
	return(CHAT_RETRY);
    default:
	ConsoleMsg(MS_E_MODEM_ABORT,
		   ret < 0 ? "Unknown": flagList[ret].name);
	return(CHAT_ABORT);
    }
}

static int
DialerWait(int fd)
{
    int ret;
    u_int32_t rcbits;

    do {
	ret = DialerInit(fd);
	if (ret != RE_OK) return(CHAT_ABORT);
	DialerSend(fd, "AutoAnswerOn");

	rcbits = ~((1<<RE_OK)|(1<<RE_RING));
	ret = DialerRecv(fd, NULL, 0, rcbits);
    } while (ret != RE_CONNECT);
    return(CHAT_DONE);
}

void
DialerClose(int fd)
{
    if (dialerPid > 0) {
	if (!kill(dialerPid, SIGQUIT)) pause();
    }
    if (fd >= 0) {
	SioIgnoreCD(fd, TRUE);
	DialerSend(fd, "HangUp");
    }
    pppInfo.l_stat &= ~LSTAT_DIAL;
}

static void
DialerQhook(int ps)
{
    extern int devFd;
    int rs;

    rs = WEXITSTATUS(ps);
    if (ISLOG(LOG_OS)) {
	Logf(LOG_OS, "dialer[%d] ", dialerPid);
	if (WIFSIGNALED(ps))
	    Logf(LOG_OS, "signal number %d\n", WTERMSIG(ps));
	else
	    Logf(LOG_OS, "%s\n", rs == CHAT_DONE ? "SUCCESS": "FAIL");
    }
    switch (rs) {
    case CHAT_DONE:
	if (devFd >= 0) {
	    SioIgnoreCD(devFd, FALSE);
	    StartDevice();
	}
	break;
    case CHAT_ABORT:
	PhaseDown();
	break;
    }
    pppInfo.l_stat &= ~LSTAT_DIAL;
    dialerPid = 0;
}

static void
DialerHandler(int sig)
{
    exit(CHAT_ABORT);
}

int
DialerStart(int fd, int argc, char *argv[])
{
    int ret, efds[2];
    enum {DM_CALL, DM_GETTY} mode=DM_CALL;

    if (argc > 1 && !strcasecmp(argv[1], "getty")) mode = DM_GETTY;
    SioIgnoreCD(fd, TRUE);
    if (!miHead) {
	if (mode == DM_GETTY) pppInfo.l_stat = LSTAT_TTY;
	return(OPEN_SUCCESS);
    }
    if ((dialerPid = MakeProcess(DialerQhook)) < 0) return(OPEN_FAILED);
    if (dialerPid != 0) {
	pppInfo.l_stat |= LSTAT_DIAL;
	/* I'm parent */
/*	PauseSelectLink(fd, TRUE);*/
    } else if (dialerPid == 0) {
	int a;
	struct numlist_s *numhp=NULL;

	numberCp = dialOpt.numhp;
	if (argc > 1 && mode == DM_CALL) {
	    struct numlist_s *nsp;
	    char *p;

	    for (a = argc - 1; a > 0; a --) {
		nsp = TALLOC(struct numlist_s);
		nsp->number = Strdup(argv[a]);
		if ((p = strrchr(nsp->number, '/')) != NULL) {
		    *p = '\0';
		    p ++;
		    nsp->retry = atoi(p);
		} else nsp->retry = 1;
		nsp->next = numhp;
		numhp = nsp;
	    }
	    if (numhp) numberCp = numhp;
	}
	/* I'm child */
	signal(SIGQUIT, DialerHandler);
	signal(SIGINT, DialerHandler);
	SetProcTitle(mode == DM_GETTY ? "(dialer-getty)":"(dialer)");
#if 0
	efds[0] = fd;
	efds[1] = -1;
	/* close files which are not used in Dialer */
	CloseSelectFds(efds);
#endif
	if (mode == DM_GETTY) {
	    exit(DialerWait(fd));
	} else if (mode == DM_CALL) {
	    dialRetry=numberCp ? numberCp->retry: -1;

	    while (numberCp) {
		dialRetry --;
		if ((ret = DialerCall(fd)) != CHAT_RETRY) exit(ret);
		usleep(dialOpt.interval * 1000000);
		if (dialRetry <= 0) {
		    numberCp = numberCp->next;
		    if (numberCp) dialRetry = numberCp->retry;
		}
	    }
	    dialRetry = -1;
	    exit(CHAT_ABORT);
	}
    }
    return(OPEN_DELAYED);
}

void
DialerSetup()
{
    static struct env_s dialenv[]={
	{"LIST", {EnvNumList}, ENV_SET, 0, 0, 0666},
	{"NUMBER", {EnvNumber}, ENV_SET|ENV_RDONLY, 0, 0, 0444},
	{"TYPE", {&dialOpt.type}, ENV_STRING, 0, 0, 0666},
	{"INTERVAL", {&dialOpt.interval}, ENV_INT32, 0, 0, 0666},
	{"TIMEOUT", {&dialOpt.timeout}, ENV_INT32, 0, 0, 0666},
	{"RATE", {&dialOpt.rate}, ENV_INT32, 0, 0, 0666},
	{NULL}
    };
    RegisterEnvs(dialenv, "DIAL", NULL);
    pppInfo.l_stat &= ~LSTAT_DIAL;
}
