#include <stdio.h>
#include <signal.h>
#include <sys/wait.h>
#include <termios.h>

#include "fppxp.h"

#include <support.h>
#include <xcio.h>
#include <ppxp.h>

static struct {
    FL_OBJECT *obj;
    const char *name, *key;
} buttons[]={
    {NULL, "Connect", "#C"},
    {NULL, "Terminal", "#T"},
    {NULL, "Bye", "#B"},
    {NULL, "Quit", "#Q"},
};

enum {
    CONNECT_BUTTON,
    TERMINAL_BUTTON,
    BYE_BUTTON,
    QUIT_BUTTON,
    MAX_BUTTONS
};

#define	BUTTON_WIDTH	70
#define	BUTTON_HEIGHT	25

static pid_t termPid;
static int autoTerm;

FL_OBJECT *autoModeObj, *openTermObj;

static void
SigChld(int sig)
{
    int ps;
    pid_t pid;

    signal(SIGCHLD, SigChld);
    pid = wait3(&ps, WNOHANG, 0);
    if (pid == termPid) autoTerm = termPid = 0;
};

void
ExecTerminal(int mode)
{
    pid_t child;
    extern char termCommand[];

    if ((child = fork()) < 0) return;
    signal(SIGCHLD, SigChld);
    if (!child) {
	char *argv[]={termCommand, "-e", PATH_PPXP, NULL};

        seteuid(getuid());
        execvp(argv[0], argv);
	{char buf[200];
	 sprintf(buf, "<%s><%s><%s>\n", argv[0], argv[1], argv[2]);
	 perror(buf);}
        exit(-1);
    }
    termPid = child;
}

void
UpdateButtons(struct pppinfo_s *old, struct pppinfo_s *new)
{
    if (old->phase != new->phase) {
	fl_set_object_label(buttons[CONNECT_BUTTON].obj,
			    (new->phase < PS_ESTABLISH)
			    ? "Connect": "Disconnect");
    }
    if (old->m_flag != new->m_flag) {
	fl_set_button(autoModeObj,
			  (new->m_flag & MFLAG_AUTO) ? 1: 0);
    }
    if (old->l_stat != new->l_stat) {
	if (new->l_stat & LSTAT_TTY) {
	    if (!termPid && fl_get_button(openTermObj)) {
		ExecTerminal(1);
		if (termPid) autoTerm = 1;
	    }
	} else if (termPid && autoTerm) {
	    kill(termPid, SIGKILL);
	}
    }
}

static void
ButtonsCallBack(FL_OBJECT *obj, long arg)
{
    int i;

    switch (arg) {
    case CONNECT_BUTTON:
	XcCommand(strcasecmp(obj->label, "disconnect") ?
	      XCMD_CONNECT: XCMD_DISCONNECT, NULL, NULL);
	break;
    case TERMINAL_BUTTON:
	if (!termPid && !autoTerm) ExecTerminal(0);
	break;
    case QUIT_BUTTON:
	XcCommand(XCMD_QUIT, NULL, NULL);
	break;
    case BYE_BUTTON:
	exit(0);
    }
}

static void
ChecksCallBack(FL_OBJECT *obj, long arg)
{
    if (obj == autoModeObj)
	XcCommand(XCMD_AUTO, NULL,
		  fl_get_button(obj) ? "on": "off", NULL);
}

void
CreateButtons(FL_FORM *form)
{
    FL_OBJECT *obj;
    int y=FRAME_GAP;
    unsigned n;

    obj = fl_add_box(FL_FRAME_BOX,
		     form->w - BUTTON_WIDTH - FRAME_GAP * 3,
		     y, BUTTON_WIDTH + FRAME_GAP * 2,
		     (BUTTON_HEIGHT + FRAME_GAP) * MAX_BUTTONS +
		     FRAME_GAP, "");
    fl_set_object_color(obj, FL_INDIANRED, FL_COL1);

    for (n = 0; n < MAX_BUTTONS; n ++) {
	y += FRAME_GAP;
	obj = fl_add_button(FL_NORMAL_BUTTON,
			    form->w - BUTTON_WIDTH - FRAME_GAP * 2,
			    y, BUTTON_WIDTH, BUTTON_HEIGHT,
			    buttons[n].name);
/*fl_set_object_boxtype(obj, FL_SHADOW_BOX);*/
	fl_set_object_callback(obj, ButtonsCallBack, n);
	fl_set_object_shortcut(obj, buttons[n].key, 1);
	buttons[n].obj = obj;
	y += BUTTON_HEIGHT;
    }
    fl_set_object_label(autoModeObj, "AUTO");
    fl_set_object_callback(autoModeObj, ChecksCallBack, 0);
    fl_set_object_label(openTermObj, "TERM");
    fl_set_object_callback(openTermObj, ChecksCallBack, 0);
}
