/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: PropertiesImpl.hxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: hbrinkm $ $Date: 2006/11/01 09:14:35 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

/*  Copyright 2005 Sun Microsystems, Inc. */

#ifndef INCLUDED_PROPERTIESIMPL_HXX
#define INCLUDED_PROPERTIESIMPL_HXX

#ifndef INCLUDED_PROPERTIES_HXX
#include <odiapi/props/Properties.hxx>
#endif

#include <set>
#include <map>
#include <list>

namespace odiapi { namespace props {

    using namespace writerfilter;

    Property::~Property() {}
    PropertyBag::~PropertyBag() {}
    PropertyPoolHandle::~PropertyPoolHandle() {}
    PropertyPool::~PropertyPool() {}
    WeakReference::~WeakReference() {}
    Subject::~Subject() {}

    class Node;

    class PropertyImpl : public Property
    {
    public:
        PropertyImpl(QName_t id);
        virtual ~PropertyImpl();

        virtual QName_t getId() const;
        virtual int getIntValue() const;
        virtual Property::Pointer_t getChild(int pos) const;
        virtual Property::Pointer_t findChild(QName_t id) const;
        virtual PropertyBag_Pointer_t getPropertyBag() const;
        virtual bool hasPropertyBag() const;
        virtual bool less(const Property& other) const;
        virtual void resolve(PropertyPoolHandle_Pointer_t realReference) const;
        
    private:
        QName_t mId;
    };

    //------------------------------------

    class IntPropertyImpl : public PropertyImpl
    {
    public:
        IntPropertyImpl(QName_t id, int value);

        virtual int getIntValue() const;
        virtual std::string getStringValue() const;
        virtual bool less(const Property& other) const;

    protected:
        IntPropertyImpl(QName_t id);
        void setValue(int value);

    private:
        int mValue;
    };

    //------------------------------------

    class StringPropertyImpl : public PropertyImpl
    {
    public:
        StringPropertyImpl(QName_t id, const std::string& value);

        virtual std::string getStringValue() const;
        virtual bool less(const Property& other) const;

    private:
        std::string mValue;
    };


    class TwipsPropertyImpl : public IntPropertyImpl
    {
    public:
        /** Constructor

        @param valueAndUnit
        As defined in the OASIS OpenDocument specification
        '<param name="pattern">-?([0-9]+(\.[0-9]*)?|\.[0-9]+)((cm)|(mm)|(in)|(pt)|(pc)|(px))</param>'
        (see http://docs.oasis-open.org/office/v1.0)

        @throws std::invalid_argument when the provided 'valueAndUnit' doesn't conform to the length
        definition of the OASIS OpenDocument specification see definition of the Relax-NG definitions 
        for custom data types in the OASIS OpenDocument specification (http://docs.oasis-open.org/office/v1.0)
        */
        TwipsPropertyImpl(QName_t id, const std::string& valueAndUnit);
	
        TwipsPropertyImpl(QName_t id, int valueInTwips);

        virtual std::string getStringValue() const;

    private:
        double valueInCm(const std::string& valueAndUnit) const;
        double getConversionFactor(const std::string& unit) const;
    };

    class CompositePropertyImpl : public PropertyImpl
    {
    public:
        CompositePropertyImpl(QName_t id, PropertyPoolHandle_Pointer_t poolHandle);
        virtual std::string getStringValue() const;
        virtual bool less(const Property& other) const;
        virtual Property::Pointer_t getChild(int pos) const;
        virtual Property::Pointer_t findChild(QName_t id) const;
        virtual PropertyBag_Pointer_t getPropertyBag() const;
      	virtual bool hasPropertyBag() const;

        /**
           For debugging purposes only
        */
        virtual Node* refersTo();

    protected:
        void setPoolHandle(PropertyPoolHandle_Pointer_t poolHandle) const;
        bool isValidPoolHandle() const;

    private:
        mutable PropertyPoolHandle_Pointer_t mPoolHandle;
    };

    /** ArrayPropertyImpl is a decorator for other properties 
        and enables the handling of properties as array elements.
    */
    class ArrayPropertyImpl : public PropertyImpl
    {
    public:
        ArrayPropertyImpl(size_t pos, Property::Pointer_t property);
	
        /** For debugging purposes
         */
        size_t getPos() const; 

        virtual int getIntValue() const;
        virtual std::string getStringValue() const;
        virtual PropertyBag_Pointer_t getPropertyBag() const;
      	virtual bool hasPropertyBag() const;

        virtual bool less(const Property& other) const;

        Property::Pointer_t getWrappedProperty();

    private:
        size_t mPos;
        Property::Pointer_t mWrappedProperty;
    };

    //------------------------------------

    class PropertyBagIteratorImpl;

    class PropertyBagImpl : public PropertyBag
    {
    public:
        PropertyBagImpl();
        virtual ~PropertyBagImpl();

        virtual size_t size() const;
        virtual void clear();
        virtual PropertyBag_Pointer_t copy() const;

        virtual void insert(Property::Pointer_t property);
        virtual Property::Pointer_t find(QName_t id) const;

        virtual void insert(int pos, Property::Pointer_t property);
        virtual Property::Pointer_t get(int pos) const;
        virtual void sort();

        virtual Iterator<Property::Pointer_t>::Pointer_t createIterator();

  private:
        void ensurePropertyNotExist(int pos);

  private:
	typedef std::map<int, Property::Pointer_t> PropertyContainer_t;
	typedef PropertyContainer_t::iterator iterator;
	typedef PropertyContainer_t::const_iterator const_iterator;
	
	PropertyContainer_t mProperties;
	
	enum { UNDEFINED, STRUCTURE, ARRAY } mIdentity;

	friend class PropertyBagIteratorImpl;        
  };

  class ReferencePropertyImpl : public CompositePropertyImpl
  {
  public:
	ReferencePropertyImpl(QName_t id, QName_t referenceProxy);
     	virtual bool less(const Property& other) const;
	    virtual void resolve(PropertyPoolHandle_Pointer_t realReference) const;
        virtual std::string getStringValue() const;
        virtual Property::Pointer_t getChild(int pos) const;
        virtual Property::Pointer_t findChild(QName_t id) const;
        virtual PropertyBag_Pointer_t getPropertyBag() const;
      	virtual bool hasPropertyBag() const;
        virtual Node* refersTo();
  private:
      QName_t referenceProxy;
  };

  //------------------------------------

  class PropertyBagIteratorImpl : public Iterator<Property::Pointer_t>
  {
  public:
	PropertyBagIteratorImpl(PropertyBagImpl& propertyBag);
	
	virtual void first();
	virtual void next();
	virtual Property::Pointer_t getCurrent() const;
	virtual bool isDone() const;

  private:
	PropertyBagImpl& mPropertyBag;
	PropertyBagImpl::iterator mIter;
  };

  /** We need a special comparison function for pointers
	  of nodes.
  */
  struct lessNodePtr : 
	  public std::binary_function<const Node*, const Node*, bool>
  {
	bool operator()(const Node* first, const Node* second) const;	
  };

  typedef std::set<Node*, lessNodePtr> NodeContainer_t;
  typedef std::set<WeakReference*> WeakReferenceContainer_t;

  class Node : public Subject
  {
  public:
	Node();
	Node(Property::Pointer_t property);
	~Node();

	void acquire();
	void release();

	void init();		
	
	/** This method is menat for debugging purposes
	 */
	bool isInUse(); 
	
	/** Subject interface
	 */
	virtual void registerWeakReference(WeakReference* ref);
	virtual void unregisterWeakReference(WeakReference* ref);

	bool hasChildren() const;

	/** References on this node from nodes other
		than parent or child nodes will be counted
	*/
	int mReferenceCount; 

	Property::Pointer_t mProperty;
	Node* mParent;
	NodeContainer_t mChildren;
	PropertyBag_Pointer_t mCachedPropertyBag;
    WeakReferenceContainer_t mListener;

#ifdef DEBUG
        static int mDebugNodeCount;
#endif
    };
  
    class PropertyPoolHandleImpl : public PropertyPoolHandle
    {
    public:
        PropertyPoolHandleImpl(Node* node);
        virtual ~PropertyPoolHandleImpl();

        virtual PropertyBag_Pointer_t getPropertyBag() const;
	
        virtual void disposing();
      
        /** This method is meant for debugging purposes
         */ 
        Node* refersTo();

    private:
        Node* mNode;
	
        friend bool operator<(const PropertyPoolHandle_Pointer_t& first, const PropertyPoolHandle_Pointer_t& second);
        friend bool operator==(const PropertyPoolHandle_Pointer_t& first, const PropertyPoolHandle_Pointer_t& second);
        friend bool operator!=(const PropertyPoolHandle_Pointer_t& first, const PropertyPoolHandle_Pointer_t& second);
    };

    class PropertyBagsIteratorImpl;

    class PropertyPoolImpl : public PropertyPool
    {
    public:
        PropertyPoolImpl(); 
	
        virtual ~PropertyPoolImpl();

        virtual Iterator<PropertyBag_Pointer_t>::Pointer_t createIterator() const;
        virtual void garbageCollection();
        virtual PropertyPoolHandle_Pointer_t insert(PropertyBag_Pointer_t propertyBag);
        virtual void dump(util::Logger* logger) const;

    public:
        size_t dbgGetNumberOfNodes();
        size_t dbgGetNumberOfUsedNodes();
        void dbgCountUsedNodes(Node* node, size_t* count);

    private:
        void unregisterNodesFromUsedNodesList();

    private:
        typedef std::list<Node*> NodeList_t;

    private:
        Node* mRootNode;
        NodeList_t mUsedNodes; // book keeping for fast dumping of PropertyBags

        friend class PropertyBagsIteratorImpl;

#ifdef DEBUG
        static int debugUseCount_;
#endif
    };

    class PropertyBagsIteratorImpl : public Iterator<PropertyBag_Pointer_t>
    {
    public:	
        PropertyBagsIteratorImpl(const PropertyPoolImpl& pool);

        virtual void first();
        virtual void next();	
        virtual PropertyBag_Pointer_t getCurrent() const;
        virtual bool isDone() const;

    private:
        void goToNextUsed();

    private:
        const PropertyPoolImpl& mPool; 
        PropertyPoolImpl::NodeList_t::const_iterator mIter;
    };

} } // namespace odiapi { namespace props {

#endif // INCLUDED_PROPERTIESIMPL_HXX
