/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: osssound.cxx,v $
 *
 *  $Revision: 1.10 $
 *
 *  last change: $Author: hr $ $Date: 2007/06/27 20:46:05 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

// MARKER(update_precomp.py): autogen include statement, do not remove
#include "precompiled_vcl.hxx"
#include <salsound.h>
#include <salimpsound.hxx>

#include <vcl/svapp.hxx>

#ifdef USE_OSS

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>
#ifdef LINUX
#include <linux/soundcard.h>
#else
#include <sys/soundcard.h>
#endif
#include <errno.h>
#include <osl/thread.h>

using namespace vcl_sal;
using namespace vos;
using namespace osl;

int						OSSSound::s_nDevice = -1;
OSSSound::OSSDataList	OSSSound::s_aDataList;
Mutex					OSSSound::s_aProtector;
OSSSound::OSSSoundList	OSSSound::s_aSounds;

BOOL OSSSound::open()
{
	MutexGuard aGuard( s_aProtector );
	if( s_nDevice == -1 )
	{
		s_nDevice = ::open( "/dev/dsp", O_WRONLY );
		if( s_nDevice == -1 )
		{
			SalDbgAssert( "ERROR: could not open /dev/dsp, errno=%d\n", errno );
			return FALSE;
		}
	}
	else
	{
		ioctl( s_nDevice, SNDCTL_DSP_RESET, 0 );
	}
	return TRUE;
}

void OSSSound::close()
{
	MutexGuard aGuard( s_aProtector );
	if( s_nDevice != -1 )
		::close( s_nDevice );
	s_nDevice = -1;
}

BOOL OSSSound::startSound( OSSData* pData )
{
	if( pData && pData->m_pSound && pData->m_pSound->m_pBuffer &&
		s_nDevice >= 0 )
	{
		if( ! strncmp( pData->m_pSound->m_pBuffer, ".snd", 4 ) )
			return startAU( pData );
		else if( ! strncmp( pData->m_pSound->m_pBuffer, "RIFF", 4 ) )
			return startRIFF( pData );
	}
	return FALSE;
}

BOOL OSSSound::startAU( OSSData* pData )
{
	char* pBuffer = pData->m_pSound->m_pBuffer;
	ULONG nDataOffset	= readBELong( pBuffer + 4 );
	ULONG nDataLen		= readBELong( pBuffer + 8 );
	ULONG nEncoding		= readBELong( pBuffer + 12 );
	ULONG nSampleRate	= readBELong( pBuffer + 16 );
	ULONG nChannels		= readBELong( pBuffer + 20 );

	if( nDataLen == 0xffffffff )
		nDataLen = pData->m_pSound->m_aStat.st_size - nDataOffset;

	pData->m_nDataLen	= nDataLen;
	pData->m_nStartPos	= nDataOffset;
	pData->m_nEndPos	= pData->m_nStartPos + pData->m_nDataLen;

	if( nChannels != 1 && nChannels != 2 )
	{
		SalDbgAssert( "%d Channels are not supported\n" );
		return FALSE;
	}

	int nSystemFormat;
	switch( nEncoding )
	{
		case 1: nSystemFormat = AFMT_MU_LAW;break;
		case 2: nSystemFormat = AFMT_U8;break;
		case 3: nSystemFormat = AFMT_S16_BE;break;
		default:
			SalDbgAssert( "au format %d unsupported\n", nEncoding );
			return FALSE;
	};

	int nRealFormat = nSystemFormat;
	if( ioctl( s_nDevice, SNDCTL_DSP_SETFMT, &nRealFormat )==-1 )
	{
		SalDbgAssert( "ERROR: ioctl SNDCTL_DSP_SETFMT failed\n" );
		return FALSE;
	}
	int nRealChannels = nChannels-1;
	if( ioctl( s_nDevice, SNDCTL_DSP_STEREO, &nRealChannels )==-1 )
	{
		SalDbgAssert( "ERROR: ioctl SNDCTL_DSP_STEREO failed\n" );
		return FALSE;
	}
	if( nRealChannels != sal::static_int_cast<int>(nChannels-1) )
	{
		SalDbgAssert( "could not set %d channels\n", nChannels );
		return FALSE;
	}

	if( ioctl( s_nDevice, SNDCTL_DSP_SPEED, &nSampleRate ) == -1 )
	{
		SalDbgAssert( "ERROR: ioctl SNDCTL_DSP_SPEED failed\n" );
		return FALSE;
	}
	SalDbgAssert( "playing %d data bytes at %d bytes in format %d quality/s on %d channels \n", pData->m_nDataLen, nSampleRate, nEncoding, nChannels );
	return TRUE;
}

BOOL OSSSound::startRIFF( OSSData* pData )
{
	int nPos = findChunk( pData, "fmt " );
	if( nPos == -1 )
		return FALSE;
	
	int nFormat		= readLEShort( pData->m_pSound->m_pBuffer + nPos + 8 );
	int nChannels	= readLEShort( pData->m_pSound->m_pBuffer + nPos + 10 );
	int nSampleRate	= readLEInt( pData->m_pSound->m_pBuffer + nPos + 12 );
	int nByteRate	= readLEInt( pData->m_pSound->m_pBuffer + nPos + 16 );
	int nAlign		= readLEShort( pData->m_pSound->m_pBuffer + nPos + 20 );
	SalDbgAssert( "format is tag = %x, channels = %d, samplesPerSec = %d, avgBytesPerSec = %d, blockAlign = %d\n", nFormat, nChannels, nSampleRate, nByteRate, nAlign );
	if( nChannels != 1 && nChannels != 2 )
	{
		SalDbgAssert( "%d Channels are not supported\n" );
		return FALSE;
	}
	
	int nBitsPerSample = 0;
	switch( nFormat )
	{
		default:
			SalDbgAssert( "unknown format\n" );
			return FALSE;
			break;
		case 1:
			nBitsPerSample = readLEShort( pData->m_pSound->m_pBuffer + nPos + 22 );
			break;
	}
	
	nPos = findChunk( pData, "data" );
	if( nPos == -1 )
	{
		SalDbgAssert( "ERROR: no \"data\" chunk found\n" );
		return FALSE;
	}
	
	pData->m_nDataLen = readLEInt( pData->m_pSound->m_pBuffer + nPos + 4 );
	pData->m_nStartPos = nPos+8;
	pData->m_nEndPos = pData->m_nStartPos + pData->m_nDataLen;
	
	int nSystemFormat;
	switch( nBitsPerSample )
	{
		case 8:  nSystemFormat = AFMT_U8; break;
		case 16: nSystemFormat = AFMT_S16_LE; break;
		default:
			SalDbgAssert( "%d bits per sample is not usable\n", nBitsPerSample );
			return FALSE;
	}
	
	int nRealFormat = nSystemFormat;
	if( ioctl( s_nDevice, SNDCTL_DSP_SETFMT, &nRealFormat )==-1 )
	{
		SalDbgAssert( "ERROR: ioctl SNDCTL_DSP_SETFMT failed\n" );
		return FALSE;
	}
	
	int nRealChannels = nChannels-1;
	if( ioctl( s_nDevice, SNDCTL_DSP_STEREO, &nRealChannels )==-1 )
	{
		SalDbgAssert( "ERROR: ioctl SNDCTL_DSP_STEREO failed\n" );
		return FALSE;
	}
	if( nRealChannels != nChannels-1 )
	{
		SalDbgAssert( "could not set %d channels\n", nChannels );
		return FALSE;
	}
	
	
	if( ioctl( s_nDevice, SNDCTL_DSP_SPEED, &nSampleRate ) == -1 )
	{
		SalDbgAssert( "ERROR: ioctl SNDCTL_DSP_SPEED failed\n" );
		return FALSE;
	}
	
	SalDbgAssert( "playing %d data bytes at %d bytes in %d bits quality/s on %d channels \n", pData->m_nDataLen, nSampleRate, nBitsPerSample, nChannels );
	return TRUE;
}

int OSSSound::findChunk( OSSData* pData, const char* pChunk )
{
	if( !pData || ! pData->m_pSound || ! pData->m_pSound->m_pBuffer )
		return -1;

	int nPos = 12;
	while( nPos < pData->m_pSound->m_aStat.st_size )
	{
		if( ! strncmp( pData->m_pSound->m_pBuffer+nPos, pChunk, 4 ) )
			return nPos;

		// get LSB int length
		nPos += readLEInt( pData->m_pSound->m_pBuffer+nPos+4 ) + 8;
	}
	return -1;
}

short int OSSSound::readLEShort( const char* pBuf )
{
	const unsigned char* pBuffer = (const unsigned char*)pBuf;
	short int nRet = *pBuffer++;
	nRet |= *pBuffer << 8;

	return nRet;
}

ULONG OSSSound::readBELong( const char* pBuf )
{
	const unsigned char* pBuffer = (const unsigned char*)pBuf;
	ULONG nRet	 = *pBuffer++ << 24;
	nRet		|= *pBuffer++ << 16;
	nRet		|= *pBuffer++ << 8;
	nRet		|= *pBuffer;

	return nRet;
}

int OSSSound::readLEInt( const char* pBuf )
{
	const unsigned char* pBuffer = (const unsigned char*)pBuf;
	int nRet = *pBuffer++;
	nRet |= (int)(*pBuffer++) << 8;
	nRet |= (int)(*pBuffer++) << 16;
	nRet |= (int)(*pBuffer++) << 24;

	return nRet;
}

static oslThread aThreadWorker= NULL;
extern "C" {
    void SAL_CALL OSSThreadWorker(void*)
    {
        OSSSound::run();
    }
}

void OSSSound::run()
{
	SalDbgAssert( "OSSWorker::run\n" );
	OSSData* pData;
    do
    {
        ResettableMutexGuard aGuard( s_aProtector );
        pData = s_aDataList.Count() ? s_aDataList.GetObject( 0 ) : NULL;
        if( ! pData )
            break;
        
        if( open() )
        {
            
            bool bDataValid = true;
            if( startSound( pData ) )
            {
                pData->m_nCurPos = pData->m_nStartPos;
                while( pData->m_nCurPos < pData->m_nEndPos )
                {
                    audio_buf_info info;
                    ioctl( s_nDevice, SNDCTL_DSP_GETOSPACE, &info );
                    if( info.bytes > 0 )
                    {
                        int nBytes = info.bytes;
                        if( pData->m_nCurPos + nBytes > pData->m_nEndPos )
                            nBytes = pData->m_nEndPos - pData->m_nCurPos;
                        nBytes = write( s_nDevice,
                                        pData->m_pSound->m_pBuffer + pData->m_nCurPos,
                                        nBytes );
                        if( nBytes > 0 )
                            pData->m_nCurPos += nBytes;
                    }
                    aGuard.clear();
                    TimeValue aVal;
                    aVal.Seconds = 0;
                    aVal.Nanosec = 20000000;
                    osl_waitThread( &aVal );
                    aGuard.reset();
                    if( s_aDataList.GetObject( 0 ) != pData )
                    {
                        ioctl( s_nDevice, SNDCTL_DSP_RESET, 0);
                        bDataValid = false;
                        break;
                    }
                }
                close();
                if( bDataValid )
                {
                    if( pData->m_pSound->m_pSalSound->m_bLoop )
                        pData->m_pSound->play();
                    else
                        Application::PostUserEvent( LINK( pData->m_pSound, OSSSound, notifyStopHdl ), NULL );
                }
            }
            else
				Application::PostUserEvent( LINK( pData->m_pSound, OSSSound, notifyErrorHdl ), (void*)SOUNDERR_INVALID_FILE );
        }
        s_aDataList.Remove( pData );
        delete pData;
    } while( pData );

    MutexGuard aGuard( s_aProtector );
    osl_destroyThread( aThreadWorker );
    aThreadWorker = NULL;
}

void OSSSound::append( OSSSound* pSound )
{
	MutexGuard aGuard( s_aProtector );
	OSSData* pNew = new OSSData;

	pNew->m_pSound		= pSound;
	pNew->m_nDataLen	= 0;
	pNew->m_nStartPos	= 0;
	pNew->m_nEndPos		= 0;
	pNew->m_nType		= 0;

	s_aDataList.Insert( pNew, LIST_APPEND );
	if( ! aThreadWorker )
        aThreadWorker = osl_createThread( OSSThreadWorker, NULL );
}

void OSSSound::remove( OSSSound* pSound )
{
	MutexGuard aGuard( s_aProtector );
	OSSData* pData;
	for( int i = s_aDataList.Count()-1; i >= 0; i-- )
	{
		pData = s_aDataList.GetObject( i );
		if( pData->m_pSound == pSound )
		{
			s_aDataList.Remove( i );
			if( i > 0 )
				delete pData;
		}
	}
}

OSSSound::OSSSound( ::X11SalSound* pSound ) :
		VSound( pSound )
{
	initBuffer();
	s_aSounds.Insert( this );
}

OSSSound::~OSSSound()
{
	stop();
 	if( m_pBuffer )
 		releaseBuffer();
	s_aSounds.Remove( this );
}

BOOL OSSSound::isValid()
{
    if( ! m_pBuffer )
        return FALSE;

    if( s_nDevice == -1 )
    {
        int nTryOpen = ::open( "/dev/dsp", O_WRONLY | O_NONBLOCK );
        if( nTryOpen == -1 )
            return FALSE;
        ::close( nTryOpen );
    }
	
	// check for valid file format
	if( ! strncmp( "RIFF", m_pBuffer, 4 ) )
	{
		OSSData aData;
		aData.m_pSound = this;

		// check for data chunk
		if( findChunk( &aData, "data" ) == -1 )
			goto failed;
		int nPos = findChunk( &aData, "fmt " );
		if( nPos == -1 )
			goto failed;
		
		int nFormat		= readLEShort( m_pBuffer + nPos + 8 );
		int nChannels	= readLEShort( m_pBuffer + nPos + 10 );
		// check channels
		if( nChannels != 1 && nChannels != 2 )
			goto failed;
		// check formats
		// playable is MS-PCM only at now
		if( nFormat != 1 )
			goto failed;
		return TRUE;
	}
	else if( ! strncmp( ".snd", m_pBuffer, 4 ) )
	{
		ULONG nEncoding		= readBELong( m_pBuffer + 12 );
		ULONG nChannels		= readBELong( m_pBuffer + 20 );

		// check for playable encodings
		if( nEncoding != 1 && nEncoding != 2 && nEncoding != 3 )
			goto failed;
		// check channels
		if( nChannels != 1 && nChannels != 2 )
			goto failed;
		return TRUE;
	}
  failed:
	releaseBuffer();
	return FALSE;
}

void OSSSound::play()
{
	if( ! m_pBuffer )
		return;

	append( this );
	m_pSalSound->m_bPlaying	= TRUE;
	m_pSalSound->m_bPaused	= FALSE;
}

void OSSSound::stop()
{
	remove( this );
	m_pSalSound->m_bPlaying	= FALSE;
	m_pSalSound->m_bPaused	= FALSE;
}

void OSSSound::pause()
{
	remove( this );
	m_pSalSound->m_bPaused = TRUE;
}

void OSSSound::cont()
{
	play();
}

IMPL_LINK( OSSSound, notifyStopHdl, void*, EMPTYARG )
{
	if( s_aSounds.GetPos( this ) != LIST_ENTRY_NOTFOUND )
	{
		m_pSalSound->changeStateStop();
	}
	return 0;
}

IMPL_LINK( OSSSound, notifyErrorHdl, void*, nError )
{
	if( s_aSounds.GetPos( this ) != LIST_ENTRY_NOTFOUND )
	{
		m_pSalSound->setError( (ULONG)nError );
	}
	return 0;
}

#endif
