/****************************************************************************
** $Id: rs_modification.cpp,v 1.44 2004/07/24 23:13:50 andrew Exp $
**
** Copyright (C) 2001-2003 RibbonSoft. All rights reserved.
**
** This file is part of the qcadlib Library project.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** Licensees holding valid qcadlib Professional Edition licenses may use 
** this file in accordance with the qcadlib Commercial License
** Agreement provided with the Software.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.ribbonsoft.com for further details.
**
** Contact info@ribbonsoft.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#include "rs_modification.h"

#include "rs_clipboard.h"
#include "rs_creation.h"
#include "rs_entity.h"
#include "rs_graphic.h"
#include "rs_information.h"
#include "rs_insert.h"
#include "rs_polyline.h"
#include "rs_text.h"



/**
 * Default constructor.
 *
 * @param container The container to which we will add
 *        entities. Usually that's an RS_Graphic entity but
 *        it can also be a polyline, text, ...
 * @param graphicView Pointer to graphic view or NULL if you don't want the
 *        any views to be updated.
 * @param handleUndo true: Handle undo functionalitiy.
 */
RS_Modification::RS_Modification(RS_EntityContainer& container,
                                 RS_GraphicView* graphicView,
                                 bool handleUndo) {
    this->container = &container;
    this->graphicView = graphicView;
    this->handleUndo = handleUndo;
    graphic = container.getGraphic();
    document = container.getDocument();
}



/**
 * Deletes all selected entities.
 */
void RS_Modification::remove() {
    if (container==NULL) {
        RS_DEBUG->print("RS_Modification::remove: no valid container",
                        RS_Debug::D_WARNING);
        return;
    }

    if (document!=NULL) {
        document->startUndoCycle();
    }

    // not safe (?)
    for (RS_Entity* e=container->firstEntity(); e!=NULL;
            e=container->nextEntity()) {

        if (e!=NULL && e->isSelected()) {
            e->setSelected(false);
            e->changeUndoState();
            if (document!=NULL) {
                document->addUndoable(e);
            }
        }
    }

    if (document!=NULL) {
        document->endUndoCycle();
    }

    graphicView->redraw();
}



/**
 * Changes the attributes of all selected 
 */
bool RS_Modification::changeAttributes(RS_AttributesData& data) {
    if (container==NULL) {
        RS_DEBUG->print("RS_Modification::changeAttributes: no valid container",
                        RS_Debug::D_WARNING);
        return false;
    }

    RS_PtrList<RS_Entity> addList;
    addList.setAutoDelete(false);

    if (document!=NULL) {
        document->startUndoCycle();
    }

    for (RS_Entity* e=container->firstEntity(); e!=NULL;
            e=container->nextEntity()) {
        //for (uint i=0; i<container->count(); ++i) {
        //RS_Entity* e = container->entityAt(i);
        if (e!=NULL && e->isSelected()) {
            RS_Entity* ec = e->clone();
            ec->setSelected(false);

            RS_Pen pen = ec->getPen(false);

            if (data.changeLayer==true) {
                ec->setLayer(data.layer);
            }

            if (data.changeColor==true) {
                pen.setColor(data.pen.getColor());
            }
            if (data.changeLineType==true) {
                pen.setLineType(data.pen.getLineType());
            }
            if (data.changeWidth==true) {
                pen.setWidth(data.pen.getWidth());
            }

            ec->setPen(pen);

            //if (data.useCurrentLayer) {
            //    ec->setLayerToActive();
            //}
            //if (data.useCurrentAttributes) {
            //    ec->setPenToActive();
            //}
            //if (ec->rtti()==RS2::EntityInsert) {
            //    ((RS_Insert*)ec)->update();
            //}
            ec->update();
            addList.append(ec);
        }
    }

    deselectOriginals(true);
    addNewEntities(addList);

    if (document!=NULL) {
        document->endUndoCycle();
    }

    if (graphicView!=NULL) {
        graphicView->redraw();
    }

    return true;
}


/**
 * Copies all selected entities from the given container to the clipboard.
 * Layers and blocks that are needed are also copied if the container is
 * or is part of an RS_Graphic.
 *
 * @param container The entity container.
 * @param ref Reference point. The entities will be moved by -ref.
 * @param cut true: cut instead of copying, false: copy
 */
void RS_Modification::copy(const RS_Vector& ref, const bool cut) {

    if (container==NULL) {
        RS_DEBUG->print("RS_Modification::copy: no valid container",
                        RS_Debug::D_WARNING);
        return;
    }

    RS_CLIPBOARD->clear();
	if (graphic!=NULL) {
    	RS_CLIPBOARD->getGraphic()->setUnit(graphic->getUnit());
	}
	else {
    	RS_CLIPBOARD->getGraphic()->setUnit(RS2::None);
	}

    // start undo cycle for the container if we're cutting
    if (cut && document!=NULL) {
        document->startUndoCycle();
    }

    // copy entities / layers / blocks
    for (RS_Entity* e=container->firstEntity(); e!=NULL;
            e=container->nextEntity()) {
        //for (uint i=0; i<container->count(); ++i) {
        //RS_Entity* e = container->entityAt(i);

        if (e!=NULL && e->isSelected()) {
            copyEntity(e, ref, cut);
        }
    }

    if (cut && document!=NULL) {
        document->endUndoCycle();
    }
}



/**
 * Copies the given entity from the given container to the clipboard.
 * Layers and blocks that are needed are also copied if the container is
 * or is part of an RS_Graphic.
 *
 * @param e The entity.
 * @param ref Reference point. The entities will be moved by -ref.
 * @param cut true: cut instead of copying, false: copy
 */
void RS_Modification::copyEntity(RS_Entity* e, const RS_Vector& ref,
                                 const bool cut) {

    if (e!=NULL && e->isSelected()) {
        // delete entity in graphic view:
        if (cut) {
            if (graphicView!=NULL) {
                graphicView->deleteEntity(e);
            }
            e->setSelected(false);
        } else {
            if (graphicView!=NULL) {
                graphicView->deleteEntity(e);
            }
            e->setSelected(false);
            if (graphicView!=NULL) {
                graphicView->drawEntity(e);
            }
        }

        // add entity to clipboard:
        RS_Entity* c = e->clone();
        c->move(-ref);
        RS_CLIPBOARD->addEntity(c);

        copyLayers(e);
        copyBlocks(e);

        // set layer to the layer clone:
        RS_Layer* l = e->getLayer();
        if (l!=NULL) {
            c->setLayer(l->getName());
        }

        // make sure all sub entities point to layers of the clipboard
        if (c->isContainer()) {
            RS_EntityContainer* ec = (RS_EntityContainer*)c;

            for (RS_Entity* e2 = ec->firstEntity(RS2::ResolveAll); e2!=NULL;
                    e2 = ec->nextEntity(RS2::ResolveAll)) {

                //RS_Entity* e2 = ec->entityAt(i);
                RS_Layer* l2 = e2->getLayer();

                if (l2!=NULL) {
                    e2->setLayer(l2->getName());
                }
            }
        }

        if (cut) {
            e->changeUndoState();
            if (document!=NULL) {
                document->addUndoable(e);
            }
        }
    }

}



/**
 * Copies all layers of the given entity to the clipboard.
 */
void RS_Modification::copyLayers(RS_Entity* e) {

    if (e==NULL) {
        return;
    }

    // add layer(s) of the entity if it's not an insert
    //  (inserts are on layer '0'):
    if (e->rtti()!=RS2::EntityInsert) {
        RS_Layer* l = e->getLayer();
        if (l!=NULL) {
            if (!RS_CLIPBOARD->hasLayer(l->getName())) {
                RS_CLIPBOARD->addLayer(l->clone());
            }
        }
    }

    // special handling of inserts:
    else {
        // insert: add layer(s) of subentities:
        RS_Block* b = ((RS_Insert*)e)->getBlockForInsert();
        if (b!=NULL) {
            for (RS_Entity* e2=b->firstEntity(); e2!=NULL;
                    e2=b->nextEntity()) {
                //for (uint i=0; i<b->count(); ++i) {
                //RS_Entity* e2 = b->entityAt(i);
                copyLayers(e2);
            }
        }
    }
}



/**
 * Copies all blocks of the given entity to the clipboard.
 */
void RS_Modification::copyBlocks(RS_Entity* e) {

    if (e==NULL) {
        return;
    }

    // add block of the entity if it's an insert
    if (e->rtti()==RS2::EntityInsert) {
        RS_Block* b = ((RS_Insert*)e)->getBlockForInsert();
        if (b!=NULL) {
            // add block of an insert:
            if (!RS_CLIPBOARD->hasBlock(b->getName())) {
                RS_CLIPBOARD->addBlock((RS_Block*)b->clone());
            }

            for (RS_Entity* e2=b->firstEntity(); e2!=NULL;
                    e2=b->nextEntity()) {
                //for (uint i=0; i<b->count(); ++i) {
                //RS_Entity* e2 = b->entityAt(i);
                copyBlocks(e2);
            }
        }
    }
}



/**
 * Pastes all entities from the clipboard into the container.
 * Layers and blocks that are needed are also copied if the container is
 * or is part of an RS_Graphic.
 *
 * @param data Paste data.
 * @param source The source from where to paste. NULL means the source
 *      is the clipboard.
 */
void RS_Modification::paste(const RS_PasteData& data, RS_Graphic* source) {

    if (/*data.asInsert==true &&*/ graphic==NULL) {
        std::cerr << "RS_Modification::paste: Can only paste as Insert, "
        "if we paste into a graphic that can hold blocks.\n";
        return;
    }

	double factor = 1.0;

    if (source==NULL) {
        source = RS_CLIPBOARD->getGraphic();

		// graphics from the clipboard need to be scaled. from the part lib not:
		RS2::Unit sourceUnit = source->getUnit();
		RS2::Unit targetUnit = graphic->getUnit();
		factor = RS_Units::convert(1.0, sourceUnit, targetUnit);
    }

    if (document!=NULL) {
        document->startUndoCycle();
    }
	

    // insert layers:
    if (graphic!=NULL) {
        RS_Layer* layer = graphic->getActiveLayer();
        for(uint i=0; i<source->countLayers(); ++i) {
            RS_Layer* l = source->layerAt(i);
            if (l!=NULL) {
                if (graphic->findLayer(l->getName())==NULL) {
                    graphic->addLayer(l->clone());
                }
            }
        }
        graphic->activateLayer(layer);
    }

    // insert blocks:
    if (graphic!=NULL) {
        for(uint i=0; i<source->countBlocks(); ++i) {
            RS_Block* b = source->blockAt(i);
            if (b!=NULL) {
                if (graphic->findBlock(b->getName())==NULL) {
                    RS_Block* bc = (RS_Block*)b->clone();
                    bc->reparent(container);
					//bc->scale(bc->getBasePoint(), RS_Vector(factor, factor));
					// scale block but don't scale inserts in block 
					//  (they already scale with their block)
        			for(uint i2=0; i2<bc->count(); ++i2) {
			            RS_Entity* e = bc->entityAt(i2);
			            if (e!=NULL && e->rtti()!=RS2::EntityInsert) {
							e->scale(bc->getBasePoint(), 
									RS_Vector(factor, factor));
						}
						else {
							RS_Vector ip = ((RS_Insert*)e)->getInsertionPoint();
							ip.scale(bc->getBasePoint(), 
								RS_Vector(factor, factor));
							((RS_Insert*)e)->setInsertionPoint(ip);
							e->update();
						}
					}

                    graphic->addBlock(bc);
                }
            }
        }
    }

    // add entities to this host (graphic or a new block)
    RS_EntityContainer* host = container;
    RS_String blockName;

    // create new block:
    if (graphic!=NULL) {
        if (data.asInsert==true) {
            RS_BlockList* blkList = graphic->getBlockList();
            if (blkList!=NULL) {
                blockName = blkList->newName(data.blockName);

                RS_Block* blk =
                    new RS_Block(graphic,
                                 RS_BlockData(blockName,
                                              RS_Vector(0.0,0.0), false));
                graphic->addBlock(blk);

                host = blk;
            }
        }
    }

    // insert entities:
    //for (uint i=0; i<((RS_EntityContainer*)source)->count(); ++i) {
    //RS_Entity* e = source->entityAt(i);
    for (RS_Entity* e=((RS_EntityContainer*)source)->firstEntity();
            e!=NULL;
            e=((RS_EntityContainer*)source)->nextEntity()) {

        if (e!=NULL) {

            RS_String layerName = "0";
            RS_Layer* layer = e->getLayer();
            if (layer!=NULL) {
                layerName = layer->getName();
            }
            RS_Entity* e2 = e->clone();
            e2->reparent(host);
            if (data.asInsert==false) {
                e2->move(data.insertionPoint);
            }
			// don't adjust insert factor - block was already adjusted to unit
			if (e2->rtti()==RS2::EntityInsert) {
				RS_Vector ip = ((RS_Insert*)e2)->getInsertionPoint();
				ip.scale(data.insertionPoint, RS_Vector(factor, factor));
				((RS_Insert*)e2)->setInsertionPoint(ip);
				e2->update();
			}
			else {
				e2->scale(data.insertionPoint, RS_Vector(factor, factor));
			}
            host->addEntity(e2);
            e2->setLayer(layerName);

            // make sure all sub entities point to layers of the container
            if (e2->isContainer()) {
                RS_EntityContainer* ec = (RS_EntityContainer*)e2;

                for (RS_Entity* e3 = ec->firstEntity(RS2::ResolveAll); e3!=NULL;
                        e3 = ec->nextEntity(RS2::ResolveAll)) {

                    //RS_Entity* e3 = ec->entityAt(i);
                    RS_Layer* l2 = e3->getLayer();
                    if (l2!=NULL) {
                        e3->setLayer(l2->getName());
                    }
                }
            }

            if (document!=NULL && data.asInsert==false) {
                document->addUndoable(e2);
            }
        }
    }

    if (data.asInsert==true) {
        RS_Insert* ins =
            new RS_Insert(container,
                          RS_InsertData(
                              blockName,
                              data.insertionPoint,
                              RS_Vector(data.factor, data.factor),
                              data.angle,
                              1,1,RS_Vector(0.0,0.0)));
        container->addEntity(ins);
        ins->setLayerToActive();
        ins->setPenToActive();

        if (document!=NULL) {
            document->addUndoable(ins);
        }
    }

    if (document!=NULL) {
        document->endUndoCycle();
    }
}


/**
 * Splits a polyline into two leaving out a gap.
 *
 * @param polyline The original polyline
 * @param e1 1st entity on which the first cutting point is.
 * @param v1 1st cutting point.
 * @param e2 2nd entity on which the first cutting point is.
 * @param v2 2nd cutting point.
 * @param polyline1 Pointer to a polyline pointer which will hold the 
 *        1st resulting new polyline. Pass NULL if you don't
 *        need those pointers.
 * @param polyline2 Pointer to a polyline pointer which will hold the 
 *        2nd resulting new polyline. Pass NULL if you don't
 *        need those pointers.
 *
 * @todo Support arcs in polylines, check for wrong parameters
 *
 * @return true
 */
bool RS_Modification::splitPolyline(RS_Polyline& polyline,
                                    RS_Entity& e1, RS_Vector v1,
                                    RS_Entity& e2, RS_Vector v2,
                                    RS_Polyline** polyline1,
                                    RS_Polyline** polyline2) const {

    if (container==NULL) {
        RS_DEBUG->print("RS_Modification::splitPolyline: no valid container",
                        RS_Debug::D_WARNING);
        return false;
    }

    RS_Entity* firstEntity = polyline.firstEntity();
    RS_Vector firstPoint(false);
    if (firstEntity->rtti()==RS2::EntityLine) {
        firstPoint = ((RS_Line*)firstEntity)->getStartpoint();
    }
    RS_Polyline* pl1 =
        new RS_Polyline(container,
                        RS_PolylineData(firstPoint, RS_Vector(0.0,0.0), 0));
    RS_Polyline* pl2 = new RS_Polyline(container);
    RS_Polyline* pl = pl1;	// Current polyline
    RS_Line* line = NULL;
    RS_Arc* arc = NULL;

    if (polyline1!=NULL) {
        *polyline1 = pl1;
    }
    if (polyline2!=NULL) {
        *polyline2 = pl2;
    }

    for (RS_Entity* e = polyline.firstEntity();
            e != NULL;
            e = polyline.nextEntity()) {

        if (e->rtti()==RS2::EntityLine) {
            line = (RS_Line*)e;
            arc = NULL;
        } else if (e->rtti()==RS2::EntityArc) {
            arc = (RS_Arc*)e;
            line = NULL;
        } else {
            line = NULL;
            arc = NULL;
        }

        if (line!=NULL /*|| arc!=NULL*/) {

            if (e==&e1 && e==&e2) {
                // Trim within a single entity:
                RS_Vector sp = line->getStartpoint();
                double dist1 = (v1-sp).magnitude();
                double dist2 = (v2-sp).magnitude();
                pl->addVertex(dist1<dist2 ? v1 : v2, 0.0);
                pl = pl2;
                pl->setStartpoint(dist1<dist2 ? v2 : v1);
                pl->addVertex(line->getEndpoint(), 0.0);
            } else if (e==&e1 || e==&e2) {
                // Trim entities:
                RS_Vector v = (e==&e1 ? v1 : v2);
                if (pl==pl1) {
                    // Trim endpoint of entity to first vector
                    pl->addVertex(v, 0.0);
                    pl = NULL;
                } else {
                    // Trim startpoint of entity to second vector
                    pl = pl2;
                    pl->setStartpoint(v);
                    pl->addVertex(line->getEndpoint(), 0.0);
                }
            } else {
                // Add entities to polylines
                if (line!=NULL && pl!=NULL) {
                    pl->addVertex(line->getEndpoint(), 0.0);
                }
            }
        }
    }

    container->addEntity(pl1);
    container->addEntity(pl2);
    //container->removeEntity(&polyline);
    polyline.changeUndoState();

    return true;
}



/**
 * Moves all selected entities with the given data for the move
 * modification.
 */
bool RS_Modification::move(RS_MoveData& data) {
    if (container==NULL) {
        RS_DEBUG->print("RS_Modification::move: no valid container",
                        RS_Debug::D_WARNING);
        return false;
    }

    RS_PtrList<RS_Entity> addList;
    addList.setAutoDelete(false);

    if (document!=NULL && handleUndo) {
        document->startUndoCycle();
    }

    // Create new entites
    for (int num=1;
            num<=data.number || (data.number==0 && num<=1);
            num++) {
        // too slow:
        //for (uint i=0; i<container->count(); ++i) {
        //RS_Entity* e = container->entityAt(i);
        for (RS_Entity* e=container->firstEntity();
                e!=NULL;
                e=container->nextEntity()) {
            if (e!=NULL && e->isSelected()) {
                RS_Entity* ec = e->clone();

                ec->move(data.offset*num);
                if (data.useCurrentLayer) {
                    ec->setLayerToActive();
                }
                if (data.useCurrentAttributes) {
                    ec->setPenToActive();
                }
                if (ec->rtti()==RS2::EntityInsert) {
                    ((RS_Insert*)ec)->update();
                }
				// since 2.0.4.0: keep selection
                ec->setSelected(true);
                addList.append(ec);
            }
        }
    }

    deselectOriginals(data.number==0);
    addNewEntities(addList);

    if (document!=NULL && handleUndo) {
        document->endUndoCycle();
    }

    if (graphicView!=NULL) {
        graphicView->redraw();
    }
    return true;
}




/**
 * Rotates all selected entities with the given data for the rotation.
 */
bool RS_Modification::rotate(RS_RotateData& data) {
    if (container==NULL) {
        RS_DEBUG->print("RS_Modification::rotate: no valid container",
                        RS_Debug::D_WARNING);
        return false;
    }

    RS_PtrList<RS_Entity> addList;
    addList.setAutoDelete(false);

    if (document!=NULL && handleUndo) {
        document->startUndoCycle();
    }

    // Create new entites
    for (int num=1;
            num<=data.number || (data.number==0 && num<=1);
            num++) {
        for (RS_Entity* e=container->firstEntity();
                e!=NULL;
                e=container->nextEntity()) {
            //for (uint i=0; i<container->count(); ++i) {
            //RS_Entity* e = container->entityAt(i);

            if (e!=NULL && e->isSelected()) {
                RS_Entity* ec = e->clone();
                ec->setSelected(false);

                ec->rotate(data.center, data.angle*num);
                if (data.useCurrentLayer) {
                    ec->setLayerToActive();
                }
                if (data.useCurrentAttributes) {
                    ec->setPenToActive();
                }
                if (ec->rtti()==RS2::EntityInsert) {
                    ((RS_Insert*)ec)->update();
                }
                addList.append(ec);
            }
        }
    }

    deselectOriginals(data.number==0);
    addNewEntities(addList);

    if (document!=NULL && handleUndo) {
        document->endUndoCycle();
    }
    if (graphicView!=NULL) {
        graphicView->redraw();
    }

    return true;
}



/**
 * Moves all selected entities with the given data for the scale
 * modification.
 */
bool RS_Modification::scale(RS_ScaleData& data) {
    if (container==NULL) {
        RS_DEBUG->print("RS_Modification::scale: no valid container",
                        RS_Debug::D_WARNING);
        return false;
    }

    RS_PtrList<RS_Entity> addList;
    addList.setAutoDelete(false);

    if (document!=NULL && handleUndo) {
        document->startUndoCycle();
    }

    // Create new entites
    for (int num=1;
            num<=data.number || (data.number==0 && num<=1);
            num++) {
        for (RS_Entity* e=container->firstEntity();
                e!=NULL;
                e=container->nextEntity()) {
            //for (uint i=0; i<container->count(); ++i) {
            //RS_Entity* e = container->entityAt(i);
            if (e!=NULL && e->isSelected()) {
                RS_Entity* ec = e->clone();
                ec->setSelected(false);

                ec->scale(data.referencePoint, RS_Math::pow(data.factor, num));
                if (data.useCurrentLayer) {
                    ec->setLayerToActive();
                }
                if (data.useCurrentAttributes) {
                    ec->setPenToActive();
                }
                if (ec->rtti()==RS2::EntityInsert) {
                    ((RS_Insert*)ec)->update();
                }
                addList.append(ec);
            }
        }
    }

    deselectOriginals(data.number==0);
    addNewEntities(addList);

    if (document!=NULL && handleUndo) {
        document->endUndoCycle();
    }

    if (graphicView!=NULL) {
        graphicView->redraw();
    }
    return true;
}



/**
 * Mirror all selected entities with the given data for the mirror
 * modification.
 */
bool RS_Modification::mirror(RS_MirrorData& data) {
    if (container==NULL) {
        RS_DEBUG->print("RS_Modification::mirror: no valid container",
                        RS_Debug::D_WARNING);
        return false;
    }

    RS_PtrList<RS_Entity> addList;
    addList.setAutoDelete(false);

    if (document!=NULL && handleUndo) {
        document->startUndoCycle();
    }

    // Create new entites
    for (int num=1;
            num<=(int)data.copy || (data.copy==false && num<=1);
            num++) {
        for (RS_Entity* e=container->firstEntity();
                e!=NULL;
                e=container->nextEntity()) {
            //for (uint i=0; i<container->count(); ++i) {
            //RS_Entity* e = container->entityAt(i);

            if (e!=NULL && e->isSelected()) {
                RS_Entity* ec = e->clone();
                ec->setSelected(false);

                ec->mirror(data.axisPoint1, data.axisPoint2);
                if (data.useCurrentLayer) {
                    ec->setLayerToActive();
                }
                if (data.useCurrentAttributes) {
                    ec->setPenToActive();
                }
                if (ec->rtti()==RS2::EntityInsert) {
                    ((RS_Insert*)ec)->update();
                }
                addList.append(ec);
            }
        }
    }

    deselectOriginals(data.copy==false);
    addNewEntities(addList);

    if (document!=NULL && handleUndo) {
        document->endUndoCycle();
    }

    if (graphicView!=NULL) {
        graphicView->redraw();
    }
    return true;
}



/**
 * Rotates entities around two centers with the given parameters.
 */
bool RS_Modification::rotate2(RS_Rotate2Data& data) {
    if (container==NULL) {
        RS_DEBUG->print("RS_Modification::rotate2: no valid container",
                        RS_Debug::D_WARNING);
        return false;
    }

    RS_PtrList<RS_Entity> addList;
    addList.setAutoDelete(false);

    if (document!=NULL && handleUndo) {
        document->startUndoCycle();
    }

    // Create new entites
    for (int num=1;
            num<=data.number || (data.number==0 && num<=1);
            num++) {

        for (RS_Entity* e=container->firstEntity();
                e!=NULL;
                e=container->nextEntity()) {
            //for (uint i=0; i<container->count(); ++i) {
            //RS_Entity* e = container->entityAt(i);

            if (e!=NULL && e->isSelected()) {
                RS_Entity* ec = e->clone();
                ec->setSelected(false);

                ec->rotate(data.center1, data.angle1*num);
                RS_Vector center2 = data.center2;
                center2.rotate(data.center1, data.angle1*num);

                ec->rotate(center2, data.angle2*num);
                if (data.useCurrentLayer) {
                    ec->setLayerToActive();
                }
                if (data.useCurrentAttributes) {
                    ec->setPenToActive();
                }
                if (ec->rtti()==RS2::EntityInsert) {
                    ((RS_Insert*)ec)->update();
                }
                addList.append(ec);
            }
        }
    }

    deselectOriginals(data.number==0);
    addNewEntities(addList);

    if (document!=NULL && handleUndo) {
        document->endUndoCycle();
    }

    if (graphicView!=NULL) {
        graphicView->redraw();
    }
    return true;
}



/**
 * Moves and rotates entities with the given parameters.
 */
bool RS_Modification::moveRotate(RS_MoveRotateData& data) {
    if (container==NULL) {
        RS_DEBUG->print("RS_Modification::moveRotate: no valid container",
                        RS_Debug::D_WARNING);
        return false;
    }

    RS_PtrList<RS_Entity> addList;
    addList.setAutoDelete(false);

    if (document!=NULL && handleUndo) {
        document->startUndoCycle();
    }

    // Create new entites
    for (int num=1;
            num<=data.number || (data.number==0 && num<=1);
            num++) {
        for (RS_Entity* e=container->firstEntity();
                e!=NULL;
                e=container->nextEntity()) {
            //for (uint i=0; i<container->count(); ++i) {
            //RS_Entity* e = container->entityAt(i);

            if (e!=NULL && e->isSelected()) {
                RS_Entity* ec = e->clone();
                ec->setSelected(false);

                ec->move(data.offset*num);
                ec->rotate(data.referencePoint + data.offset*num,
                           data.angle*num);
                if (data.useCurrentLayer) {
                    ec->setLayerToActive();
                }
                if (data.useCurrentAttributes) {
                    ec->setPenToActive();
                }
                if (ec->rtti()==RS2::EntityInsert) {
                    ((RS_Insert*)ec)->update();
                }
                addList.append(ec);
            }
        }
    }

    deselectOriginals(data.number==0);
    addNewEntities(addList);

    if (document!=NULL && handleUndo) {
        document->endUndoCycle();
    }
    if (graphicView!=NULL) {
        graphicView->redraw();
    }

    return true;
}



/**
 * Deselects all selected entities and removes them if remove is true;
 *
 * @param remove true: Remove entites.
 */
void RS_Modification::deselectOriginals(bool remove) {
    for (RS_Entity* e=container->firstEntity();
            e!=NULL;
            e=container->nextEntity()) {

        //for (uint i=0; i<container->count(); ++i) {
        //RS_Entity* e = container->entityAt(i);

        if (e!=NULL) {
            bool selected = false;

            /*
                  if (e->isAtomic()) {
                      RS_AtomicEntity* ae = (RS_AtomicEntity*)e;
                      if (ae->isStartpointSelected() ||
                              ae->isEndpointSelected()) {

                          selected = true;
                      }
                  }
            */

            if (e->isSelected()) {
                selected = true;
            }

            if (selected) {
                e->setSelected(false);
                if (remove) {
                    //if (graphicView!=NULL) {
                    //    graphicView->deleteEntity(e);
                    //}
                    e->changeUndoState();
                    if (document!=NULL && handleUndo) {
                        document->addUndoable(e);
                    }
                } else {
                    //if (graphicView!=NULL) {
                    //    graphicView->drawEntity(e);
                    //}
                }
            }
        }
    }
}



/**
 * Adds the given entities to the container and draws the entities if 
 * there's a graphic view available.
 *
 * @param addList Entities to add.
 */
void RS_Modification::addNewEntities(RS_PtrList<RS_Entity>& addList) {
    for (RS_Entity* e=addList.first();
            e!=NULL;
            e=addList.next()) {
        if (e!=NULL) {
            container->addEntity(e);
            if (document!=NULL && handleUndo) {
                document->addUndoable(e);
            }
            //if (graphicView!=NULL) {
            //    graphicView->drawEntity(e);
            //}
        }
    }
}



/**
 * Trims or extends the given trimEntity to the intersection point of the
 * trimEntity and the limitEntity.
 *
 * @param trimCoord Coordinate which defines which endpoint of the 
 *   trim entity to trim.
 * @param trimEntity Entity which will be trimmed.
 * @param limitCoord Coordinate which defines the intersection to which the
 *    trim entity will be trimmed.
 * @param limitEntity Entity to which the trim entity will be trimmed.
 * @param both true: Trim both entities. false: trim trimEntity only.
 */
bool RS_Modification::trim(const RS_Vector& trimCoord,
                           RS_AtomicEntity* trimEntity,
                           const RS_Vector& limitCoord,
                           RS_Entity* limitEntity,
                           bool both) {

    if (trimEntity==NULL || limitEntity==NULL) {
        std::cerr << "RS_Modification::trim: At least one entity is NULL.\n";
        return false;
    }

    if (both && !limitEntity->isAtomic()) {
        std::cerr << "RS_Modification::trim: limitEntity is not atomic.\n";
    }

    RS_VectorSolutions sol;
    if (limitEntity->isAtomic()) {
        //std::cout << "limitEntity->isAtomic\n";
        // intersection(s) of the two entities:
        sol = RS_Information::getIntersection(trimEntity, limitEntity, false);
    } else if (limitEntity->isContainer()) {
        //std::cout << "limitEntity->isContainer\n";
        RS_EntityContainer* ec = (RS_EntityContainer*)limitEntity;

        sol.alloc(1000);
        int i=0;

        for (RS_Entity* e=ec->firstEntity(RS2::ResolveAll); e!=NULL;
                e=ec->nextEntity(RS2::ResolveAll)) {
            //for (int i=0; i<container->count(); ++i) {
            //    RS_Entity* e = container->entityAt(i);

            if (e!=NULL) {

                RS_VectorSolutions s2 = RS_Information::getIntersection(trimEntity,
                                        e, false);

                //std::cout << "sol: " << s2 << "\n";

                if (s2.hasValid()) {
                    //std::cout << "  found\n";
                    for (int k=0; k<s2.getNumber(); ++k) {
                        //std::cout << "  k: " << k << "\n";
                        if (i<1000 && s2.get(k).valid) {
                            //std::cout << "  found valid\n";
                            if (e->isPointOnEntity(s2.get(k), 1.0e-4)) {
                                //std::cout << "  on entity\n";
                                sol.set(i++, s2.get(k));
                                //std::cout << "  vector: " << s2.get(k) << "\n";
                            }
                        }
                    }
                    //break;
                }
            }
        }
    }

    if (sol.hasValid()==false) {
        //std::cerr << "RS_Modification::trim: No intersection possible between "
        //"given entities.\n";
        return false;
    }

    RS_AtomicEntity* trimmed1 = NULL;
    RS_AtomicEntity* trimmed2 = NULL;

    // remove trim entity from view:
    if (trimEntity->rtti()==RS2::EntityCircle) {
        // convert a circle into a trimmable arc
        RS_Circle* c = (RS_Circle*)trimEntity;
        double am = c->getCenter().angleTo(trimCoord);
        RS_ArcData d(c->getCenter(),
                     c->getRadius(),
                     RS_Math::correctAngle(am-M_PI/2),
                     RS_Math::correctAngle(am+M_PI/2), false);
        trimmed1 = new RS_Arc(trimEntity->getParent(), d);
    } else {
        trimmed1 = (RS_AtomicEntity*)trimEntity->clone();
        trimmed1->setHighlighted(false);
    }
    if (graphicView!=NULL) {
        graphicView->deleteEntity(trimEntity);
    }

    // remove limit entity from view:
    if (both) {
        trimmed2 = (RS_AtomicEntity*)limitEntity->clone();
        trimmed2->setHighlighted(false);
        if (graphicView!=NULL) {
            graphicView->deleteEntity(limitEntity);
        }
    }

    // trim trim entity
    int ind = 0;
    RS_Vector is = sol.getClosest(trimCoord, NULL, &ind);
    RS_Vector is2 = sol.get(ind==0 ? 1 : 0);

    //std::cout << "is: " << is << "\n";
    //std::cout << "is2: " << is2 << "\n";

    RS2::Ending ending = trimmed1->getTrimPoint(trimCoord, is);

    switch (ending) {
    case RS2::EndingStart:
        trimmed1->moveStartpoint(is);
        if (trimEntity->rtti()==RS2::EntityCircle) {
            trimmed1->moveEndpoint(is2);
        }
        break;
    case RS2::EndingEnd:
        trimmed1->moveEndpoint(is);
        if (trimEntity->rtti()==RS2::EntityCircle) {
            trimmed1->moveStartpoint(is2);
        }
        break;
    default:
        break;
    }

    // trim limit entity:
    if (both) {
        RS_Vector is = sol.getClosest(limitCoord);

        RS2::Ending ending = trimmed2->getTrimPoint(limitCoord, is);

        switch (ending) {
        case RS2::EndingStart:
            trimmed2->moveStartpoint(is);
            break;
        case RS2::EndingEnd:
            trimmed2->moveEndpoint(is);
            break;
        default:
            break;
        }
    }

    // add new trimmed trim entity:
    container->addEntity(trimmed1);
    if (graphicView!=NULL) {
        graphicView->drawEntity(trimmed1);
    }

    // add new trimmed limit entity:
    if (both) {
        container->addEntity(trimmed2);
        if (graphicView!=NULL) {
            graphicView->drawEntity(trimmed2);
        }
    }

    if (document!=NULL && handleUndo) {
        document->startUndoCycle();
        document->addUndoable(trimmed1);
        trimEntity->setUndoState(true);
        document->addUndoable(trimEntity);
        if (both) {
            document->addUndoable(trimmed2);
            limitEntity->setUndoState(true);
            document->addUndoable(limitEntity);
        }
        document->endUndoCycle();
    }

    return true;
}



/**
 * Trims or extends the given trimEntity by the given amount.
 *
 * @param trimCoord Coordinate which defines which endpoint of the 
 *   trim entity to trim.
 * @param trimEntity Entity which will be trimmed.
 * @param dist Amount to trim by.
 */
bool RS_Modification::trimAmount(const RS_Vector& trimCoord,
                                 RS_AtomicEntity* trimEntity,
                                 double dist) {

    if (trimEntity==NULL) {
        std::cerr << "RS_Modification::trimAmount: Entity is NULL.\n";
        return false;
    }

    RS_AtomicEntity* trimmed = NULL;

    // remove trim entity:
    trimmed = (RS_AtomicEntity*)trimEntity->clone();
    if (graphicView!=NULL) {
        graphicView->deleteEntity(trimEntity);
    }

    // trim trim entity
    RS_Vector is = trimmed->getNearestDist(-dist, trimCoord);
    if (trimCoord.distanceTo(trimmed->getStartpoint()) <
            trimCoord.distanceTo(trimmed->getEndpoint())) {
        trimmed->moveStartpoint(is);
    } else {
        trimmed->moveEndpoint(is);
    }

    // add new trimmed trim entity:
    container->addEntity(trimmed);

    //std::cout << "coord: " << trimCoord << "\n";
    //std::cout << "dist: " << dist << "\n";
    //std::cout << "trimmed: " << *trimmed << "\n";

    if (graphicView!=NULL) {
        graphicView->drawEntity(trimmed);
    }

    if (document!=NULL && handleUndo) {
        document->startUndoCycle();
        document->addUndoable(trimmed);
        trimEntity->setUndoState(true);
        document->addUndoable(trimEntity);
        document->endUndoCycle();
    }

    return true;
}



/**
 * Cuts the given entity at the given point.
 */
bool RS_Modification::cut(const RS_Vector& cutCoord,
                          RS_AtomicEntity* cutEntity) {

    if (cutEntity==NULL) {
        std::cerr << "RS_Modification::cut: Entity is NULL.\n";
        return false;
    }

    if (!cutCoord.valid) {
        std::cerr << "RS_Modification::cut: Point invalid.\n";
        return false;
    }

    // cut point is at endpoint of entity:
    if (cutCoord.distanceTo(cutEntity->getStartpoint())<1.0e-6 ||
            cutCoord.distanceTo(cutEntity->getEndpoint())<1.0e-6) {
        std::cerr << "RS_Modification::cut: Cutting point on endpoint.\n";
        return false;
    }

    // delete cut entity on the screen:
    if (graphicView!=NULL) {
        graphicView->deleteEntity(cutEntity);
    }

    RS_AtomicEntity* cut1 = NULL;
    RS_AtomicEntity* cut2 = NULL;

    // create new two halves:
    if (cutEntity->rtti()==RS2::EntityCircle) {
        RS_Circle* c = (RS_Circle*)cutEntity;
        cut1 = new RS_Arc(cutEntity->getParent(),
                          RS_ArcData(c->getCenter(),
                                     c->getRadius(),
                                     0.0,0.0, false));
        cut1->setPen(cutEntity->getPen());
        cut1->setLayer(cutEntity->getLayer());
        cut2 = NULL;

        cut1->moveEndpoint(cutCoord);
        cut1->moveStartpoint(cutCoord);
    } else {
        cut1 = (RS_AtomicEntity*)cutEntity->clone();
        cut2 = (RS_AtomicEntity*)cutEntity->clone();

        cut1->moveEndpoint(cutCoord);
        cut2->moveStartpoint(cutCoord);
    }

    // add new cut entity:
    container->addEntity(cut1);
    if (cut2!=NULL) {
        container->addEntity(cut2);
    }

    if (graphicView!=NULL) {
        graphicView->drawEntity(cut1);
        if (cut2!=NULL) {
            graphicView->drawEntity(cut2);
        }
    }

    if (document!=NULL && handleUndo) {
        document->startUndoCycle();
        document->addUndoable(cut1);
        if (cut2!=NULL) {
            document->addUndoable(cut2);
        }
        cutEntity->setUndoState(true);
        document->addUndoable(cutEntity);
        document->endUndoCycle();
    }

    return true;
}



/**
 * Stretching.
 */
bool RS_Modification::stretch(const RS_Vector& firstCorner,
                              const RS_Vector& secondCorner,
                              const RS_Vector& offset) {

    if (!offset.valid) {
        std::cerr << "RS_Modification::stretch: Offset invalid.\n";
        return false;
    }

    RS_PtrList<RS_Entity> addList;
    addList.setAutoDelete(false);

    if (document!=NULL && handleUndo) {
        document->startUndoCycle();
    }

    // Create new entites
    for (RS_Entity* e=container->firstEntity();
            e!=NULL;
            e=container->nextEntity()) {
        //for (int i=0; i<container->count(); ++i) {
        //    RS_Entity* e = container->entityAt(i);

        if (e!=NULL &&
                e->isVisible() &&
                !e->isLocked() &&
                (e->isInWindow(firstCorner, secondCorner) ||
                 e->hasEndpointsWithinWindow(firstCorner, secondCorner))) {

            RS_Entity* ec = e->clone();
            ec->stretch(firstCorner, secondCorner, offset);
            addList.append(ec);
            e->setSelected(true);
        }
    }

    deselectOriginals(true);
    addNewEntities(addList);

    if (document!=NULL && handleUndo) {
        document->endUndoCycle();
    }

    if (graphicView!=NULL) {
        graphicView->redraw();
    }
    return true;
}



/**
 * Bevels a corner.
 *
 * @param coord1 Mouse coordinate to specify direction from intersection.
 * @param entity1 First entity of the corner.
 * @param coord2 Mouse coordinate to specify direction from intersection.
 * @param entity2 Second entity of the corner.
 * @param data Lengths and trim flag.
 */
bool RS_Modification::bevel(const RS_Vector& coord1, RS_AtomicEntity* entity1,
                            const RS_Vector& coord2, RS_AtomicEntity* entity2,
                            RS_BevelData& data) {

    if (entity1==NULL || entity2==NULL) {
        std::cerr << "RS_Modification::bevel: At least one entity is NULL.\n";
        return false;
    }

    //std::cout << "bevel: get inters\n";

    RS_VectorSolutions sol =
        RS_Information::getIntersection(entity1, entity2, false);

    if (sol.getNumber()==0) {
        //std::cerr << "RS_Modification::bevel: No intersection possible between "
        //"given entities.\n";
        return false;
    }

    RS_AtomicEntity* trimmed1 = NULL;
    RS_AtomicEntity* trimmed2 = NULL;

    //std::cout << "bevel: remove trim entities\n";

    trimmed1 = (RS_AtomicEntity*)entity1->clone();
    trimmed2 = (RS_AtomicEntity*)entity2->clone();

    // remove trim entity:
    if (data.trim==true) {
        if (graphicView!=NULL) {
            graphicView->deleteEntity(entity1);
            graphicView->deleteEntity(entity2);
        }
    }

    //std::cout << "bevel: trim to interes\n";

    // trim entities to intersection
	bool start1 = false;
    RS_Vector is = sol.getClosest(coord2);
    RS2::Ending ending1 = trimmed1->getTrimPoint(coord1, is);
    switch (ending1) {
    case RS2::EndingStart:
        trimmed1->moveStartpoint(is);
		start1 = true;
        break;
    case RS2::EndingEnd:
        trimmed1->moveEndpoint(is);
		start1 = false;
        break;
    default:
        break;
    }

	bool start2 = false;
    is = sol.getClosest(coord1);
    RS2::Ending ending2 = trimmed2->getTrimPoint(coord2, is);
    switch (ending2) {
    case RS2::EndingStart:
        trimmed2->moveStartpoint(is);
		start2 = true;
        break;
    case RS2::EndingEnd:
        trimmed2->moveEndpoint(is);
		start2 = false;
        break;
    default:
        break;
    }

    //std::cout << "bevel: find bevel points: ";

    // find definitive bevel points
    RS_Vector bp1 = trimmed1->getNearestDist(data.length1, start1);
    RS_Vector bp2 = trimmed2->getNearestDist(data.length2, start2);

    //std::cout << " " << bp1 << " " << bp2 << "\n";

    //std::cout << "bevel: final trim\n";

    // final trim:
    if (data.trim==true) {
        switch (ending1) {
        case RS2::EndingStart:
            trimmed1->moveStartpoint(bp1);
            break;
        case RS2::EndingEnd:
            trimmed1->moveEndpoint(bp1);
            break;
        default:
            break;
        }

        switch (ending2) {
        case RS2::EndingStart:
            trimmed2->moveStartpoint(bp2);
            break;
        case RS2::EndingEnd:
            trimmed2->moveEndpoint(bp2);
            break;
        default:
            break;
        }
    }

    // add new trimmed entities:
    if (data.trim==true) {
        container->addEntity(trimmed1);
        container->addEntity(trimmed2);
        if (graphicView!=NULL) {
            graphicView->drawEntity(trimmed1);
            graphicView->drawEntity(trimmed2);
        }
    }

    // add bevel:
    RS_Line* bevel = new RS_Line(container, RS_LineData(bp1, bp2));
    container->addEntity(bevel);

    if (graphicView!=NULL) {
        graphicView->drawEntity(bevel);
    }

    if (document!=NULL && handleUndo) {
        document->startUndoCycle();

        if (data.trim==true) {
            document->addUndoable(trimmed1);
            entity1->setUndoState(true);
            document->addUndoable(entity1);

            document->addUndoable(trimmed2);
            entity2->setUndoState(true);
            document->addUndoable(entity2);
        }

        document->addUndoable(bevel);

        document->endUndoCycle();
    }

    if (data.trim==false) {
        delete trimmed1;
        delete trimmed2;
    }

    return true;

}



/**
 * Rounds a corner.
 *
 * @param coord Mouse coordinate to specify the rounding.
 * @param entity1 First entity of the corner.
 * @param entity2 Second entity of the corner.
 * @param data Radius and trim flag.
 */
bool RS_Modification::round(const RS_Vector& coord,
                            const RS_Vector& coord1,
                            RS_AtomicEntity* entity1,
                            const RS_Vector& coord2,
                            RS_AtomicEntity* entity2,
                            RS_RoundData& data) {

    if (entity1==NULL || entity2==NULL) {
        std::cerr << "RS_Modification::round: At least one entity is NULL.\n";
        return false;
    }

    //std::cout << "round: get inters\n";

    // create 2 tmp parallels
    //RS_Graphic tmp;
    RS_Creation creation(NULL, NULL);
    RS_Entity* par1 = creation.createParallel(coord, data.radius, 1, entity1);
    RS_Entity* par2 = creation.createParallel(coord, data.radius, 1, entity2);

    //par1->reparent(container);
    //par2->reparent(container);
    //container->addEntity(par1);
    //container->addEntity(par2);

    //if (graphicView!=NULL) {
    //    graphicView->drawEntity(par1);
    //    graphicView->drawEntity(par2);
    //}

    RS_VectorSolutions sol2 =
        RS_Information::getIntersection(entity1, entity2, false);

    RS_VectorSolutions sol =
        RS_Information::getIntersection(par1, par2, false);

    if (sol.getNumber()==0) {
        //std::cerr << "RS_Modification::round: No intersection possible between "
        //"given entities.\n";
        return false;
    }

    // there might be two intersections: choose the closest:
    RS_Vector is = sol.getClosest(coord);
    RS_Vector p1 = entity1->getNearestPointOnEntity(is, false);
    RS_Vector p2 = entity2->getNearestPointOnEntity(is, false);
    double ang1 = is.angleTo(p1);
    double ang2 = is.angleTo(p2);
    bool reversed = (RS_Math::getAngleDifference(ang1, ang2)>M_PI);

    RS_Arc* arc = new RS_Arc(container,
                             RS_ArcData(is,
                                        data.radius,
                                        ang1, ang2,
                                        reversed));


    RS_AtomicEntity* trimmed1 = NULL;
    RS_AtomicEntity* trimmed2 = NULL;

    if (data.trim) {

        //std::cout << "round: remove trim entities\n";

        trimmed1 = (RS_AtomicEntity*)entity1->clone();
        trimmed2 = (RS_AtomicEntity*)entity2->clone();

        // remove trim entity:
        if (graphicView!=NULL) {
            graphicView->deleteEntity(entity1);
            graphicView->deleteEntity(entity2);
        }

        //std::cout << "round: trim to interes\n";

        // trim entities to intersection
        RS_Vector is2 = sol2.getClosest(coord2);
        RS2::Ending ending1 = trimmed1->getTrimPoint(coord1, is2);
        switch (ending1) {
        case RS2::EndingStart:
            trimmed1->moveStartpoint(p1);
            break;
        case RS2::EndingEnd:
            trimmed1->moveEndpoint(p1);
            break;
        default:
            break;
        }

        is2 = sol2.getClosest(coord1);
        RS2::Ending ending2 = trimmed2->getTrimPoint(coord2, is2);
        switch (ending2) {
        case RS2::EndingStart:
            trimmed2->moveStartpoint(p2);
            break;
        case RS2::EndingEnd:
            trimmed2->moveEndpoint(p2);
            break;
        default:
            break;
        }

        // add new trimmed entities:
        container->addEntity(trimmed1);
        container->addEntity(trimmed2);
        if (graphicView!=NULL) {
            graphicView->drawEntity(trimmed1);
            graphicView->drawEntity(trimmed2);
        }
    }

    // add rounding:
    container->addEntity(arc);
    if (graphicView!=NULL) {
        graphicView->drawEntity(arc);
    }

    if (document!=NULL && handleUndo) {
        document->startUndoCycle();

        if (data.trim==true) {
            document->addUndoable(trimmed1);
            entity1->setUndoState(true);
            document->addUndoable(entity1);

            document->addUndoable(trimmed2);
            entity2->setUndoState(true);
            document->addUndoable(entity2);
        }

        document->addUndoable(arc);

        document->endUndoCycle();
    }

    delete par1;
    delete par2;

    return true;

}



/**
 * Removes the selected entity containers and adds the entities in them as
 * new single entities.
 */
bool RS_Modification::explode() {

    if (container==NULL) {
        RS_DEBUG->print("RS_Modification::explode: no valid container"
                        " for addinge entities",
                        RS_Debug::D_WARNING);
        return false;
    }

    RS_PtrList<RS_Entity> addList;
    addList.setAutoDelete(false);

    if (document!=NULL && handleUndo) {
        document->startUndoCycle();
    }

    for (RS_Entity* e=container->firstEntity();
            e!=NULL;
            e=container->nextEntity()) {
        //for (uint i=0; i<container->count(); ++i) {
        //RS_Entity* e = container->entityAt(i);

        if (e!=NULL && e->isSelected()) {
            if (e->isContainer()) {

                // add entities from container:
                RS_EntityContainer* ec = (RS_EntityContainer*)e;
                //ec->setSelected(false);

                // iterate and explode container:
                //for (uint i2=0; i2<ec->count(); ++i2) {
                //    RS_Entity* e2 = ec->entityAt(i2);
                RS2::ResolveLevel rl;
                bool resolvePen;
                bool resolveLayer;

                switch (ec->rtti()) {
                case RS2::EntityText:
                case RS2::EntityHatch:
                case RS2::EntityPolyline:
                    rl = RS2::ResolveAll;
                    resolveLayer = true;
                    resolvePen = false;
                    break;

                case RS2::EntityInsert:
                    resolvePen = false;
                    resolveLayer = false;
                    rl = RS2::ResolveNone;
                    break;

                case RS2::EntityDimAligned:
                case RS2::EntityDimLinear:
                case RS2::EntityDimRadial:
                case RS2::EntityDimDiametric:
                case RS2::EntityDimAngular:
                case RS2::EntityDimLeader:
                    rl = RS2::ResolveNone;
                    resolveLayer = true;
                    resolvePen = false;
                    break;

                default:
                    rl = RS2::ResolveAll;
                    resolveLayer = true;
                    resolvePen = false;
                    break;
                }

                for (RS_Entity* e2 = ec->firstEntity(rl); e2!=NULL;
                        e2 = ec->nextEntity(rl)) {

                    if (e2!=NULL) {

                        RS_Entity* clone = e2->clone();
                        clone->setSelected(false);
                        clone->reparent(container);

                        if (resolveLayer) {
                            clone->setLayer(ec->getLayer());
                        } else {
                            clone->setLayer(e2->getLayer());
                        }

                        clone->setPen(ec->getPen(resolvePen));

                        addList.append(clone);

                        clone->update();
                    }
                }
            } else {
                e->setSelected(false);
            }
        }
    }

    deselectOriginals(true);
    addNewEntities(addList);

    if (document!=NULL && handleUndo) {
        document->endUndoCycle();
    }

    if (graphicView!=NULL) {
        graphicView->redraw();
    }

    return true;
}



bool RS_Modification::explodeTextIntoLetters() {
    if (container==NULL) {
        RS_DEBUG->print("RS_Modification::explodeTextIntoLetters: no valid container"
                        " for addinge entities",
                        RS_Debug::D_WARNING);
        return false;
    }

    RS_PtrList<RS_Entity> addList;
    addList.setAutoDelete(false);

    if (document!=NULL && handleUndo) {
        document->startUndoCycle();
    }

    for (RS_Entity* e=container->firstEntity();
            e!=NULL;
            e=container->nextEntity()) {
        if (e!=NULL && e->isSelected()) {
            if (e->rtti()==RS2::EntityText) {
                // add letters of text:
                RS_Text* text = (RS_Text*)e;
                explodeTextIntoLetters(text, addList);
            } else {
                e->setSelected(false);
            }
        }
    }

    deselectOriginals(true);
    addNewEntities(addList);

    if (document!=NULL && handleUndo) {
        document->endUndoCycle();
    }

    if (graphicView!=NULL) {
        graphicView->redraw();
    }

    return true;
}



bool RS_Modification::explodeTextIntoLetters(RS_Text* text, RS_PtrList<RS_Entity>& addList) {

    if (text==NULL) {
        return false;
    }

    // iterate though lines:
    for (RS_Entity* e2 = text->firstEntity(); e2!=NULL;
            e2 = text->nextEntity()) {

        if (e2==NULL) {
            break;
        }


        // text lines:
        if (e2->rtti()==RS2::EntityContainer) {

            RS_EntityContainer* line = (RS_EntityContainer*)e2;

            // iterate though letters:
            for (RS_Entity* e3 = line->firstEntity(); e3!=NULL;
                    e3 = line->nextEntity()) {
        
				if (e3==NULL) {
            		break;
        		}

				// super / sub texts:
				if (e3->rtti()==RS2::EntityText) {
					explodeTextIntoLetters((RS_Text*)e3, addList);
				}
				
				// normal letters:
                else if (e3->rtti()==RS2::EntityInsert) {

                    RS_Insert* letter = (RS_Insert*)e3;

                    RS_Text* tl = new RS_Text(
                                      container,
                                      RS_TextData(letter->getInsertionPoint(),
                                                  text->getHeight(),
                                                  100.0,
                                                  RS2::VAlignBottom, RS2::HAlignLeft,
                                                  RS2::LeftToRight, RS2::Exact,
                                                  1.0,
                                                  letter->getName(),
                                                  text->getStyle(),
                                                  letter->getAngle(),
                                                  RS2::Update));

                    tl->setLayer(text->getLayer());
                    tl->setPen(text->getPen());

                    addList.append(tl);
                    tl->update();
                }
            }
        }
    }

    return true;
}



/**
 * Moves all reference points of selected entities with the given data.
 */
bool RS_Modification::moveRef(RS_MoveRefData& data) {
    if (container==NULL) {
        RS_DEBUG->print("RS_Modification::moveRef: no valid container",
                        RS_Debug::D_WARNING);
        return false;
    }

    RS_PtrList<RS_Entity> addList;
    addList.setAutoDelete(false);

    if (document!=NULL && handleUndo) {
        document->startUndoCycle();
    }

    // Create new entites
    for (RS_Entity* e=container->firstEntity();
            e!=NULL;
            e=container->nextEntity()) {
        if (e!=NULL && e->isSelected()) {
            RS_Entity* ec = e->clone();

            ec->moveRef(data.ref, data.offset);
			// since 2.0.4.0: keep it selected
            ec->setSelected(true);
            addList.append(ec);
        }
    }

    deselectOriginals(true);
    addNewEntities(addList);

    if (document!=NULL && handleUndo) {
        document->endUndoCycle();
    }

    if (graphicView!=NULL) {
        graphicView->redraw();
    }
    return true;
}

// EOF
