/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package mx4j.tools.adaptor.rmi;

import java.io.IOException;
import java.rmi.MarshalledObject;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.InvalidAttributeValueException;
import javax.management.MBeanException;
import javax.management.MBeanRegistrationException;
import javax.management.MBeanServer;
import javax.management.NotCompliantMBeanException;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.management.ReflectionException;

import mx4j.server.ChainedMBeanServer;

/**
 * A decorator for MBeanServer that takes care of unmarshalling RMI arguments that are known to the MBean
 * but not to the adaptor.
 *
 * @author <a href="mailto:biorn_steedom@users.sourceforge.net">Simone Bordet</a>
 * @version $Revision: 1.1 $
 */
public class UnmarshallingMBeanServer extends ChainedMBeanServer
{
	public UnmarshallingMBeanServer(MBeanServer server)
	{
		super(server);
	}

	public ObjectInstance createMBean(String className, ObjectName objectName, Object[] args, String[] parameters)
			throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException, MBeanException, NotCompliantMBeanException
	{
		try
		{
			Object[] unmarshalled = unmarshalObjects(args);
			return super.createMBean(className, objectName, unmarshalled, parameters);
		}
		catch (ClassNotFoundException x)
		{
			throw new ReflectionException(x);
		}
		catch (IOException x)
		{
			throw new ReflectionException(x);
		}
	}

	public ObjectInstance createMBean(String className, ObjectName objectName, ObjectName loaderName, Object[] args, String[] parameters)
			throws ReflectionException, InstanceAlreadyExistsException, MBeanRegistrationException, MBeanException, NotCompliantMBeanException, InstanceNotFoundException
	{
		try
		{
			Object[] unmarshalled = unmarshalObjects(args);
			return super.createMBean(className, objectName, loaderName, unmarshalled, parameters);
		}
		catch (ClassNotFoundException x)
		{
			throw new ReflectionException(x);
		}
		catch (IOException x)
		{
			throw new ReflectionException(x);
		}
	}

	public void setAttribute(ObjectName objectName, Attribute attribute)
			throws InstanceNotFoundException, AttributeNotFoundException, InvalidAttributeValueException, MBeanException, ReflectionException
	{
		try
		{
			attribute = unmarshalAttribute(attribute);
			super.setAttribute(objectName, attribute);
		}
		catch (ClassNotFoundException x)
		{
			throw new ReflectionException(x);
		}
		catch (IOException x)
		{
			throw new ReflectionException(x);
		}
	}

	public AttributeList setAttributes(ObjectName objectName, AttributeList attributes)
			throws InstanceNotFoundException, ReflectionException
	{
		try
		{
			AttributeList list = new AttributeList();
			for (int i = 0; i < attributes.size(); ++i)
			{
				Attribute attribute = (Attribute)attributes.get(i);
				attribute = unmarshalAttribute(attribute);
				list.add(attribute);
			}
			return super.setAttributes(objectName, list);
		}
		catch (ClassNotFoundException x)
		{
			throw new ReflectionException(x);
		}
		catch (IOException x)
		{
			throw new ReflectionException(x);
		}
	}

	public Object invoke(ObjectName objectName, String methodName, Object[] args, String[] parameters)
			throws InstanceNotFoundException, MBeanException, ReflectionException
	{
		try
		{
			Object[] unmarshalled = unmarshalObjects(args);
			return super.invoke(objectName, methodName, unmarshalled, parameters);
		}
		catch (ClassNotFoundException x)
		{
			throw new ReflectionException(x);
		}
		catch (IOException x)
		{
			throw new ReflectionException(x);
		}
	}

	private Object[] unmarshalObjects(Object[] args) throws ClassNotFoundException, IOException
	{
		// The connector sent us Objects and MarshalledObjects; here we have the context classloader set to the
		// MBean's classloader, so the unmarshalling should complete with no problems.

		if (args == null) return null;

		Object[] unmarshalled = new Object[args.length];
		for (int i = 0; i < unmarshalled.length; ++i)
		{
			Object arg = args[i];
			unmarshalled[i] = unmarshalObject(arg);
		}
		return unmarshalled;
	}

	private Object unmarshalObject(Object obj) throws ClassNotFoundException, IOException
	{
		if (obj instanceof MarshalledObject)
		{
			return ((MarshalledObject)obj).get();
		}
		else
		{
			return obj;
		}
	}

	private Attribute unmarshalAttribute(Attribute attribute) throws ClassNotFoundException, IOException
	{
		Object value = attribute.getValue();
		Object unmarshalled = unmarshalObject(value);
		if (value != unmarshalled)
		{
			attribute = new Attribute(attribute.getName(), unmarshalled);
		}
		return attribute;
	}
}
