/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package test.javax.management.monitor;

import java.util.Set;

import junit.framework.TestCase;
import javax.management.MBeanServerFactory;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.management.ObjectInstance;
import javax.management.Attribute;
import javax.management.NotificationListener;
import javax.management.Notification;
import javax.management.monitor.CounterMonitor;
import javax.management.monitor.MonitorNotification;
import mx4j.log.*;

/**
 * Class CounterMonitorTest, tests the Counter monitoring service
 *
 * @author <a href="mailto:tibu@users.sourceforge.net">Carlos Quiroz</a>
 * @version $Revision: 1.7 $
 */
public class CounterMonitorTest extends TestCase {
    private MBeanServer server;

    /**
     * Dummy test interface
     */
    public interface TestClassMBean {
        public Integer getInt();

        public void setInt(Integer number);
    }

    public static class TestClass implements TestClassMBean {
        private Integer number;

        public TestClass(Integer number) {
            this.number = number;
        }

        public Integer getInt() {
            return  number;
        }

        public void setInt(Integer number) {
            this.number = number;
        }
    }

    public static class MonitorListener implements NotificationListener {
        MonitorNotification lastNotification;

        public void handleNotification(Notification notification, Object handback) {
            if (notification instanceof MonitorNotification) {
                lastNotification = (MonitorNotification)notification;
            }
        }

        public MonitorNotification getLastNotification() {
            return lastNotification;
        }
    }

    /**
     * Constructor requested by the JUnit framework
     */
    public CounterMonitorTest() {
        super("CounterMonitor Test");
    }

    /**
     * Constructor requested by the JUnit framework
     */
    public CounterMonitorTest(String name) {
        super(name);
    }

    protected void setUp() throws Exception {
        super.setUp();
        server = MBeanServerFactory.createMBeanServer("CounterMonitor");
    }

    /**
     * test JMX 1.1 style invocation patterns
     */
    public void testBasics() throws Exception {
        ObjectName monitorName = new ObjectName("CounterMonitor:name=monitor1");
        try {
            server.createMBean("javax.management.monitor.CounterMonitor", monitorName, null);
            // test defaults that apply to all mbeans
            assertEquals(new Integer(0), server.getAttribute(monitorName, "InitThreshold"));
            assertEquals(new Integer(0), server.getAttribute(monitorName, "Modulus"));
            assertEquals(new Integer(0), server.getAttribute(monitorName, "Offset"));
            assertEquals(Boolean.FALSE, server.getAttribute(monitorName, "DifferenceMode"));
            assertEquals(Boolean.FALSE, server.getAttribute(monitorName, "Notify"));
            // set tests
            boolean exception = false;
            try {
                server.setAttribute(monitorName, new Attribute("Threshold",  new Integer(-1)));
            }
            catch (Exception e) {
                exception = true;
            }
            assertTrue(exception);
            exception = false;

            // note:  this pattern shouldn't work with JMX 1.2 style stuff. Setting
            // the threshold without specifying an object to monitor creates weird results.
            // setThreshold has been replaced by setInitThreshold... should we return the
            // initThreshold?  Or should we be consistent and say that getThreshold() is
            // equivalent to getThreshold(objectNames.get(0))?
            server.setAttribute(monitorName, new Attribute("InitThreshold",  new Integer(4)));
            assertEquals(new Integer(4), server.getAttribute(monitorName, "InitThreshold"));
            try {
                server.setAttribute(monitorName, new Attribute("Modulus",  new Integer(-1)));
            }
            catch (Exception e) {
                exception = true;
            }
            assertTrue(exception);
            exception = false;
            server.setAttribute(monitorName, new Attribute("Modulus", new Integer(3)));
            assertEquals(new Integer(3), server.getAttribute(monitorName, "Modulus"));
            try {
                server.setAttribute(monitorName, new Attribute("Offset",  new Integer(-1)));
            }
            catch (Exception e) {
                exception = true;
            }
            assertTrue(exception);
            exception = false;
            server.setAttribute(monitorName, new Attribute("Offset", new Integer(6)));
            assertEquals(new Integer(6), server.getAttribute(monitorName, "Offset"));
            server.setAttribute(monitorName, new Attribute("DifferenceMode", Boolean.TRUE));
            assertEquals(Boolean.TRUE, server.getAttribute(monitorName, "DifferenceMode"));
            server.setAttribute(monitorName, new Attribute("Notify", Boolean.TRUE));
            assertEquals(Boolean.TRUE, server.getAttribute(monitorName, "Notify"));
        }
        finally {
            try{
                server.unregisterMBean(monitorName);
            }catch(Exception ignore){
            }
        }
    }

    /**
     * this test looks a little flaky, just tests for valid invocation of
     * the methods -markmcbride
     */
    public void testStartStopSequence() throws Exception {
        ObjectName name = new ObjectName("CounterMonitor:name=monitorTarget");
        ObjectName monitorName = new ObjectName("CounterMonitor:name=monitor1");
        try {
            TestClass a = new TestClass(new Integer(10));
            server.registerMBean(a, name);
            server.createMBean("javax.management.monitor.CounterMonitor", monitorName, null);
            server.setAttribute(monitorName, new Attribute("ObservedObject", name));
            server.setAttribute(monitorName, new Attribute("ObservedAttribute", "Int"));
            server.setAttribute(monitorName, new Attribute("GranularityPeriod", new Long(1000L)));
            server.setAttribute(monitorName, new Attribute("Threshold", new Integer(11)));
            server.setAttribute(monitorName, new Attribute("Modulus", new Integer(0)));
            server.setAttribute(monitorName, new Attribute("Offset", new Integer(0)));
            server.setAttribute(monitorName, new Attribute("Notify", Boolean.TRUE));
            server.setAttribute(monitorName, new Attribute("DifferenceMode", Boolean.TRUE));
            MonitorListener listener = new MonitorListener();
            server.addNotificationListener(monitorName, listener, null, null);
            server.invoke(monitorName, "start", new Object[] {}, new String[] {});
            server.invoke(monitorName, "stop", new Object[] {}, new String[] {});
            server.invoke(monitorName, "start", new Object[] {}, new String[] {});
        }
        finally {
            server.unregisterMBean(name);
            server.unregisterMBean(monitorName);
        }
    }

    /**
     * test JMX 1.1 style invocation patterns
     */
    public void testLegacyMonitor() throws Exception {
        ObjectName name = new ObjectName("CounterMonitor:name=monitorTarget");
        ObjectName monitorName = new ObjectName("CounterMonitor:name=monitor1");
        try {
            TestClass a = new TestClass(new Integer(10));
            server.registerMBean(a, name);
            server.createMBean("javax.management.monitor.CounterMonitor", monitorName, null);
            server.setAttribute(monitorName, new Attribute("ObservedObject", name));
            server.setAttribute(monitorName, new Attribute("ObservedAttribute", "Int"));
            server.setAttribute(monitorName, new Attribute("GranularityPeriod", new Long(1000L)));
            server.setAttribute(monitorName, new Attribute("Threshold", new Integer(11)));
            server.setAttribute(monitorName, new Attribute("Modulus", new Integer(0)));
            server.setAttribute(monitorName, new Attribute("Offset", new Integer(0)));
            server.setAttribute(monitorName, new Attribute("Notify", Boolean.TRUE));
            server.setAttribute(monitorName, new Attribute("DifferenceMode", Boolean.TRUE));
            MonitorListener listener = new MonitorListener();
            server.addNotificationListener(monitorName, listener, null, null);
            server.invoke(monitorName, "start", new Object[] {}, new String[] {});

            Thread.sleep(1200);

            assertEquals(new Integer(0),(Integer)server.getAttribute(monitorName, "DerivedGauge"));
            assertTrue(listener.getLastNotification() == null);

            Thread.sleep(1000);

            assertEquals(new Integer(0),(Integer)server.getAttribute(monitorName, "DerivedGauge"));
            assertTrue(listener.getLastNotification() == null);
            server.setAttribute(name, new Attribute("Int", new Integer(12)));

            Thread.sleep(1000);

            assertEquals(new Integer(2),(Integer)server.getAttribute(monitorName, "DerivedGauge"));
            assertEquals(MonitorNotification.THRESHOLD_VALUE_EXCEEDED, listener.getLastNotification().getType());
            long sequence = listener.getLastNotification().getSequenceNumber();

            Thread.sleep(1000);

            assertEquals(new Integer(0),(Integer)server.getAttribute(monitorName, "DerivedGauge"));
            // repeat last notification. i.e. no new notifications
            assertEquals(MonitorNotification.THRESHOLD_VALUE_EXCEEDED, listener.getLastNotification().getType());
            assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
            server.setAttribute(name, new Attribute("Int", new Integer(5)));

            Thread.sleep(1000);

            assertEquals(new Integer(-7),(Integer)server.getAttribute(monitorName, "DerivedGauge"));
            // still the same as before
            assertEquals(MonitorNotification.THRESHOLD_VALUE_EXCEEDED, listener.getLastNotification().getType());
            assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
            server.setAttribute(name, new Attribute("Int", new Integer(15)));

            Thread.sleep(1000);

            // new notification
            assertEquals(MonitorNotification.THRESHOLD_VALUE_EXCEEDED, listener.getLastNotification().getType());
            assertTrue(sequence != listener.getLastNotification().getSequenceNumber());
            sequence = listener.getLastNotification().getSequenceNumber();
            // test with offset
            server.setAttribute(monitorName, new Attribute("Offset", new Integer(5)));

            Thread.sleep(1000);

            // check new threshold
            assertEquals(MonitorNotification.THRESHOLD_VALUE_EXCEEDED, listener.getLastNotification().getType());
            assertTrue(sequence != listener.getLastNotification().getSequenceNumber());
            assertEquals(new Integer(0),(Integer)server.getAttribute(monitorName, "DerivedGauge"));
            assertEquals(new Integer(16), (Integer)server.getAttribute(monitorName, "Threshold"));
            sequence = listener.getLastNotification().getSequenceNumber();
            server.setAttribute(name, new Attribute("Int", new Integer(34)));

            Thread.sleep(1000);

            assertEquals(MonitorNotification.THRESHOLD_VALUE_EXCEEDED, listener.getLastNotification().getType());
            assertTrue(sequence != listener.getLastNotification().getSequenceNumber());
            sequence = listener.getLastNotification().getSequenceNumber();
            assertEquals(new Integer(19), (Integer)server.getAttribute(monitorName, "DerivedGauge"));
            // check new threshold
            assertEquals(new Integer(36), (Integer)server.getAttribute(monitorName, "Threshold"));
            server.setAttribute(monitorName, new Attribute("Modulus", new Integer(50)));

            Thread.sleep(1000);

            assertEquals(MonitorNotification.THRESHOLD_VALUE_EXCEEDED, listener.getLastNotification().getType());
            assertEquals(new Integer(0),(Integer)server.getAttribute(monitorName, "DerivedGauge"));
            assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
            server.setAttribute(name, new Attribute("Int", new Integer(60)));

            Thread.sleep(1000);

            assertEquals(new Integer(10), (Integer)server.getAttribute(monitorName, "DerivedGauge"));
            // check threshold rollover
            assertEquals(new Integer(11), (Integer)server.getAttribute(monitorName, "Threshold"));
        }
        finally {
            server.unregisterMBean(name);
            server.unregisterMBean(monitorName);
        }
    }

    /**
     * test JMX 1.2 style invocation patterns
     */
    public void testMonitor() throws Exception {
        ObjectName name = new ObjectName("CounterMonitor:name=monitorTarget1");
        ObjectName name2 = new ObjectName("CounterMonitor:name=monitorTarget2");
        ObjectName monitorName = new ObjectName("CounterMonitor:name=monitor1");
        try {
            TestClass a = new TestClass(new Integer(10));
            server.registerMBean(a, name);
            TestClass b = new TestClass(new Integer(2));
            server.registerMBean(b, name2);
            server.createMBean("javax.management.monitor.CounterMonitor", monitorName, null);
            //server.setAttribute(monitorName, new Attribute("ObservedObject", name));
            server.invoke(monitorName, "addObservedObject", new Object[] {name}, new String [] {"javax.management.ObjectName"});
            server.invoke(monitorName, "addObservedObject", new Object[] {name2}, new String [] {"javax.management.ObjectName"});
            server.setAttribute(monitorName, new Attribute("ObservedAttribute", "Int"));
            server.setAttribute(monitorName, new Attribute("GranularityPeriod", new Long(1000L)));
            server.setAttribute(monitorName, new Attribute("Threshold", new Integer(11)));
            server.setAttribute(monitorName, new Attribute("Modulus", new Integer(0)));
            server.setAttribute(monitorName, new Attribute("Offset", new Integer(0)));
            server.setAttribute(monitorName, new Attribute("Notify", Boolean.TRUE));
            server.setAttribute(monitorName, new Attribute("DifferenceMode", Boolean.TRUE));
            MonitorListener listener = new MonitorListener();
            server.addNotificationListener(monitorName, listener, null, null);
            server.invoke(monitorName, "start", new Object[] {}, new String[] {});

            Thread.sleep(1200);

            assertEquals(new Integer(0),(Integer)server.invoke(monitorName, "getDerivedGauge", new Object[]{name},new String[]{"javax.management.ObjectName"}));

            assertTrue(listener.getLastNotification() == null);

            Thread.sleep(1000);

            assertEquals(new Integer(0),(Integer)server.invoke(monitorName, "getDerivedGauge", new Object[]{name},new String[]{"javax.management.ObjectName"}));
            assertEquals(new Integer(0),(Integer)server.invoke(monitorName, "getDerivedGauge", new Object[]{name2},new String[]{"javax.management.ObjectName"}));
            assertTrue(listener.getLastNotification() == null);
            server.setAttribute(name, new Attribute("Int", new Integer(12)));

            Thread.sleep(1000);

            assertEquals(new Integer(2),(Integer)server.invoke(monitorName, "getDerivedGauge", new Object[]{name},new String[]{"javax.management.ObjectName"}));
            assertEquals(new Integer(0),(Integer)server.invoke(monitorName, "getDerivedGauge", new Object[]{name2},new String[]{"javax.management.ObjectName"}));
            assertEquals(MonitorNotification.THRESHOLD_VALUE_EXCEEDED, listener.getLastNotification().getType());


            server.setAttribute(name2, new Attribute("Int", new Integer(12)));
            Thread.sleep(1000);

            long sequence = listener.getLastNotification().getSequenceNumber();
            assertEquals(new Integer(0),(Integer)server.invoke(monitorName, "getDerivedGauge", new Object[]{name},new String[]{"javax.management.ObjectName"}));
            assertEquals(new Integer(10),(Integer)server.invoke(monitorName, "getDerivedGauge", new Object[]{name2},new String[]{"javax.management.ObjectName"}));
            assertEquals(MonitorNotification.THRESHOLD_VALUE_EXCEEDED, listener.getLastNotification().getType());
            Thread.sleep(1000);

            assertEquals(new Integer(0),(Integer)server.invoke(monitorName, "getDerivedGauge", new Object[]{name},new String[]{"javax.management.ObjectName"}));
            // repeat last notification. i.e. no new notifications
            assertEquals(MonitorNotification.THRESHOLD_VALUE_EXCEEDED, listener.getLastNotification().getType());
            assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
            server.setAttribute(name, new Attribute("Int", new Integer(5)));
            server.setAttribute(name2, new Attribute("Int", new Integer(3)));

            Thread.sleep(1000);

            assertEquals(new Integer(-7),(Integer)server.invoke(monitorName, "getDerivedGauge", new Object[]{name},new String[]{"javax.management.ObjectName"}));
            assertEquals(new Integer(-9),(Integer)server.invoke(monitorName, "getDerivedGauge", new Object[]{name2},new String[]{"javax.management.ObjectName"}));
            // still the same as before
            assertEquals(MonitorNotification.THRESHOLD_VALUE_EXCEEDED, listener.getLastNotification().getType());
            assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
            server.setAttribute(name, new Attribute("Int", new Integer(15)));

            Thread.sleep(1000);

            // new notification
            assertEquals(MonitorNotification.THRESHOLD_VALUE_EXCEEDED, listener.getLastNotification().getType());
            assertTrue(sequence != listener.getLastNotification().getSequenceNumber());
            sequence = listener.getLastNotification().getSequenceNumber();
            // test with offset
            server.setAttribute(monitorName, new Attribute("Offset", new Integer(5)));

            Thread.sleep(1000);

            // check new threshold
            assertEquals(MonitorNotification.THRESHOLD_VALUE_EXCEEDED, listener.getLastNotification().getType());
            assertTrue(sequence != listener.getLastNotification().getSequenceNumber());
            assertEquals(new Integer(0),(Integer)server.invoke(monitorName, "getDerivedGauge", new Object[]{name},new String[]{"javax.management.ObjectName"}));
            assertEquals(new Integer(16), (Integer)server.invoke(monitorName, "getThreshold", new Object[]{name},new String[]{"javax.management.ObjectName"}));
            assertEquals(new Integer(0),(Integer)server.invoke(monitorName, "getDerivedGauge", new Object[]{name2},new String[]{"javax.management.ObjectName"}));
            assertEquals(new Integer(11), (Integer)server.invoke(monitorName, "getThreshold", new Object[]{name2},new String[]{"javax.management.ObjectName"}));
            sequence = listener.getLastNotification().getSequenceNumber();
            server.setAttribute(name, new Attribute("Int", new Integer(34)));
            server.setAttribute(name2, new Attribute("Int", new Integer(32)));

            Thread.sleep(1000);

            assertEquals(MonitorNotification.THRESHOLD_VALUE_EXCEEDED, listener.getLastNotification().getType());
            assertTrue(sequence != listener.getLastNotification().getSequenceNumber());
            sequence = listener.getLastNotification().getSequenceNumber();
            assertEquals(new Integer(19), (Integer)server.invoke(monitorName, "getDerivedGauge", new Object[]{name},new String[]{"javax.management.ObjectName"}));
            assertEquals(new Integer(29), (Integer)server.invoke(monitorName, "getDerivedGauge", new Object[]{name2},new String[]{"javax.management.ObjectName"}));
            // check new threshold
            assertEquals(new Integer(36), (Integer)server.invoke(monitorName, "getThreshold", new Object[]{name},new String[]{"javax.management.ObjectName"}));
            assertEquals(new Integer(36), (Integer)server.invoke(monitorName, "getThreshold", new Object[]{name2},new String[]{"javax.management.ObjectName"}));
            server.setAttribute(monitorName, new Attribute("Modulus", new Integer(50)));

            Thread.sleep(1000);

            assertEquals(MonitorNotification.THRESHOLD_VALUE_EXCEEDED, listener.getLastNotification().getType());
            assertEquals(new Integer(0),(Integer)server.invoke(monitorName, "getDerivedGauge", new Object[]{name},new String[]{"javax.management.ObjectName"}));
            assertEquals(sequence, listener.getLastNotification().getSequenceNumber());
            server.setAttribute(name, new Attribute("Int", new Integer(60)));
            server.setAttribute(name2, new Attribute("Int", new Integer(62)));

            Thread.sleep(1000);

            assertEquals(new Integer(10), (Integer)server.invoke(monitorName, "getDerivedGauge", new Object[]{name},new String[]{"javax.management.ObjectName"}));
            assertEquals(new Integer(12), (Integer)server.invoke(monitorName, "getDerivedGauge", new Object[]{name2},new String[]{"javax.management.ObjectName"}));
            // check threshold rollover
            assertEquals(new Integer(11), (Integer)server.invoke(monitorName, "getThreshold", new Object[]{name},new String[]{"javax.management.ObjectName"}));
            assertEquals(new Integer(11), (Integer)server.invoke(monitorName, "getThreshold", new Object[]{name2},new String[]{"javax.management.ObjectName"}));
        }
        finally {
            server.unregisterMBean(name);
            server.unregisterMBean(name2);
            server.unregisterMBean(monitorName);
        }
    }

    public void testAddRemove() throws Exception{
        ObjectName name1 = new ObjectName("CounterMonitor:name=monitorTarget1");
        ObjectName name2 = new ObjectName("CounterMonitor:name=monitorTarget2");
        ObjectName name3 = new ObjectName("CounterMonitor:name=monitorTarget3");
        ObjectName monitorName = new ObjectName("CounterMonitor:name=monitor1");
        try {
            TestClass a = new TestClass(new Integer(10));
            server.registerMBean(a, name1);
            TestClass b = new TestClass(new Integer(2));
            server.registerMBean(b, name2);
            TestClass c = new TestClass(new Integer(2));
            server.registerMBean(c, name3);
            server.createMBean("javax.management.monitor.CounterMonitor", monitorName, null);
            //server.setAttribute(monitorName, new Attribute("ObservedObject", name));
            server.invoke(monitorName, "addObservedObject", new Object[] {name1}, new String [] {"javax.management.ObjectName"});
            server.setAttribute(monitorName, new Attribute("ObservedAttribute", "Int"));
            server.setAttribute(monitorName, new Attribute("GranularityPeriod", new Long(1000L)));
            server.setAttribute(monitorName, new Attribute("Threshold", new Integer(11)));
            server.setAttribute(monitorName, new Attribute("Modulus", new Integer(0)));
            server.setAttribute(monitorName, new Attribute("Offset", new Integer(0)));
            server.setAttribute(monitorName, new Attribute("Notify", Boolean.TRUE));
            server.setAttribute(monitorName, new Attribute("DifferenceMode", Boolean.TRUE));
            MonitorListener listener = new MonitorListener();
            server.addNotificationListener(monitorName, listener, null, null);
            server.invoke(monitorName, "start", new Object[] {}, new String[] {});

            // should have one observed object
            ObjectName [] names = (ObjectName[])server.getAttribute(monitorName, "ObservedObjects");
            assertTrue(names.length == 1);
            assertEquals(names[0], name1);

            // add another, make sure we have two
            server.invoke(monitorName, "addObservedObject", new Object[] {name2}, new String [] {"javax.management.ObjectName"});
            names = (ObjectName[])server.getAttribute(monitorName, "ObservedObjects");
            assertTrue(names.length == 2);
            assertEquals(names[0],name1);
            assertEquals(names[1],name2);

            // remove one, make sure we're back to one
            server.invoke(monitorName, "removeObservedObject", new Object[] {name2}, new String [] {"javax.management.ObjectName"});
            names = (ObjectName[])server.getAttribute(monitorName, "ObservedObjects");
            assertTrue(names.length == 1);
            assertEquals(names[0],name1);

            // now add back a different one
            server.invoke(monitorName, "addObservedObject", new Object[] {name3}, new String [] {"javax.management.ObjectName"});
            names = (ObjectName[])server.getAttribute(monitorName, "ObservedObjects");
            assertTrue(names.length == 2);
            assertEquals(names[0],name1);
            assertEquals(names[1],name3);

            // try to remove one that doesn't exist. Should NOT get IllegalArgumentException since JMX 1.2
            boolean errorFound = false;
            try{
                server.invoke(monitorName, "removeObservedObject", new Object[] {name2}, new String [] {"javax.management.ObjectName"});
            }catch(javax.management.RuntimeMBeanException mbe){
                assertTrue(mbe.getTargetException() instanceof IllegalArgumentException);
                errorFound = true;
            }
            assertFalse(errorFound);

            // try to add one that already exists. Should NOT get IllegalArgumentException since JMX 1.2
            errorFound = false;
            try{
                server.invoke(monitorName, "addObservedObject", new Object[] {name3}, new String [] {"javax.management.ObjectName"});
            }catch(javax.management.RuntimeMBeanException mbe){
                assertTrue(mbe.getTargetException() instanceof IllegalArgumentException);
                errorFound = true;
            }
            assertFalse(errorFound);
        }finally {
            server.unregisterMBean(name1);
            server.unregisterMBean(name2);
            server.unregisterMBean(name3);
            server.unregisterMBean(monitorName);
        }
    }
}

