/*
 * groach
 * GNOME applet version main module.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <signal.h>

#include <gnome.h>
#include <gdk/gdkx.h>
#include <applet-widget.h>

#include "groach.h"

#include "types.h"
#include "amain.h"
#include "theme-search.h"
#include "properties.h"

#include <X11/Xatom.h>/* for XA_WINDOW */
#include "vroot.h"/* file is taken from gnome-libs-1.0.54 */


/* Private function declarations */
static GroachData* groach_data_new(AppletWidget *applet, GroachProp *groach_prop);
static void groach_data_delete(GroachData *groach_data);

static void create_widgets(GroachData *groach_data);

static gint save_session(GtkWidget *widget, char *cfgpath, char *globcfgpath, GroachData *groach_data); 
static void run_button_cb(GtkButton *button, GroachData *groach_data);
static void stop_button_cb(GtkButton *button, GroachData *groach_data);
static void speed_changed_cb(GtkAdjustment *adj, GroachData *groach_data);
static void about_cb(AppletWidget *widget, gpointer data);

static void sighup_handler(int sig);


int
main(int argc, char *argv[])
{
	GtkWidget *applet;
	GroachData *groach_data;
	GroachProp *groach_prop;

	srand((int)(getpid() + time(NULL)));
	signal(SIGHUP, sighup_handler);/* ridiculous... see the description below. */
	
	/* Initialize the i18n stuff */
	bindtextdomain(PACKAGE, GNOMELOCALEDIR);
	textdomain(PACKAGE);
	
	/* Intialize, this will basically set up the applet, corba and
	   call gnome_init */
	applet_widget_init(APPLET_NAME, VERSION, argc, argv, NULL, 0, NULL);

	/* create a new applet_widget */
	applet = applet_widget_new(APPLET_NAME);
	/* in the rare case that the communication with the panel
	   failed, error out */
	if (!applet)
		g_error("Can't create applet!\n");

	groach_prop = properties_new();
	properties_load(groach_prop, APPLET_WIDGET(applet)->privcfgpath);
	
	groach_data = groach_data_new(APPLET_WIDGET(applet), groach_prop);
	create_widgets(groach_data);

	gtk_signal_connect(GTK_OBJECT(applet), "delete_event",
					   GTK_SIGNAL_FUNC(gtk_main_quit), NULL);
	gtk_signal_connect(GTK_OBJECT(applet), "save_session",
					   GTK_SIGNAL_FUNC(save_session), groach_data);
	
	/* special corba main loop */
	applet_widget_gtk_main();

	properties_delete(groach_prop);
	groach_data_delete(groach_data);
	
	return 0;
}


/* ---The followings are private functions--- */
static GroachData*
groach_data_new(AppletWidget *applet, GroachProp *groach_prop)
{
	GroachData *groach_data;
	GroTheme *theme;
	const char *theme_name;
	char *theme_dirname = NULL;
	
	groach_data = g_new(GroachData, 1);
	groach_data->applet = applet;
	groach_data->groach_prop = groach_prop;

	/* For Virtual Root Window handling (see vroot.h),
	   I don't use GDK_ROOT_WINDOW() nor GDK_ROOT_PARENT(). */
	groach_data->root_win = gdk_window_foreign_new(DefaultRootWindow(GDK_DISPLAY()));
	groach_data->controller = gro_controller_new(groach_data->root_win, NULL);

	groach_data->theme_search = theme_search_new(NULL);

	theme_name = groach_prop->theme_name;
	if (theme_name && theme_name[0]) {
		theme_dirname = theme_search_find_path(groach_data->theme_search, theme_name);
		if (theme_dirname == NULL) {
			g_warning(_("Can't find theme: %s\n"), theme_name);
		}
	}

	if (theme_dirname == NULL) {
		theme_dirname = theme_search_find_path(groach_data->theme_search, THEME_DEFAULT);
		if (theme_dirname == NULL) {
			g_error(_("Can't find theme: %s\n"), THEME_DEFAULT);
		}
	}
	theme = gro_theme_load(theme_dirname, THEME_LOAD_NORMAL);
	if (theme == NULL) {
		g_print(_("Can't load theme: %s\n"), theme_dirname);
		/* todo: open dialog to select theme */
		gtk_main_quit();
	}
	g_free(theme_dirname);
	gro_controller_attach_theme(groach_data->controller, theme);
	gro_controller_add_gmoves(groach_data->controller, NUM_GMOVES_DEFAULT);

	gro_controller_run(groach_data->controller);

	return groach_data;
}

static void
groach_data_delete(GroachData *groach_data)
{
	theme_search_delete(groach_data->theme_search);

	if (gro_controller_is_running(groach_data->controller) == TRUE) {
		gro_controller_stop(groach_data->controller);
	}
	gro_controller_detach_theme(groach_data->controller);
	gro_controller_delete(groach_data->controller);

	g_free(groach_data);

	gdk_window_unref(groach_data->root_win);
}

static void
create_widgets(GroachData *groach_data)
{
	AppletWidget *applet = groach_data->applet;
	GtkStyle *style;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *run_button;
	GtkWidget *stop_button;
	GtkWidget *stock;
	GtkObject *adj;
	GtkWidget *speed_scale;

	style = gtk_widget_get_style(GTK_WIDGET(applet));
	vbox = gtk_vbox_new(FALSE, 0);
	applet_widget_add(applet, vbox);

	hbox = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, FALSE);

	run_button = gtk_button_new();
	stock = gnome_stock_new_with_icon(GNOME_STOCK_PIXMAP_FORWARD);
	gtk_container_add(GTK_CONTAINER(run_button), stock);
	gtk_signal_connect(GTK_OBJECT(run_button), "clicked",
					   GTK_SIGNAL_FUNC(run_button_cb), groach_data);
	gtk_box_pack_start(GTK_BOX(hbox), run_button, FALSE, FALSE, FALSE);

	stop_button = gtk_button_new();
	stock = gnome_stock_new_with_icon(GNOME_STOCK_PIXMAP_STOP);
	gtk_container_add(GTK_CONTAINER(stop_button), stock);
	gtk_signal_connect(GTK_OBJECT(stop_button), "clicked",
					   GTK_SIGNAL_FUNC(stop_button_cb), groach_data);
	gtk_box_pack_start(GTK_BOX(hbox), stop_button, FALSE, FALSE, FALSE);

	adj = gtk_adjustment_new(50, 10, 200, 1, 10, 10);
	speed_scale = gtk_hscale_new(GTK_ADJUSTMENT(adj));
	gtk_scale_set_draw_value(GTK_SCALE(speed_scale), FALSE);
	gtk_range_set_update_policy(GTK_RANGE(speed_scale),
								GTK_UPDATE_DISCONTINUOUS);
	gtk_signal_connect(adj, "value_changed",
					   GTK_SIGNAL_FUNC(speed_changed_cb), groach_data);
	gtk_box_pack_start(GTK_BOX(vbox), speed_scale, FALSE, FALSE, FALSE);

	applet_widget_set_tooltip(applet, _("Groach controller"));

	applet_widget_register_stock_callback(applet, "properties",
										  GNOME_STOCK_MENU_PROP,
										  _("Properties..."),
										  property_cb, groach_data);

	applet_widget_register_stock_callback(applet, "about",
										  GNOME_STOCK_MENU_ABOUT,
										  _("About..."), about_cb, NULL);

	gtk_widget_show_all(GTK_WIDGET(applet));
}


/* Signal handlers */
static gint
save_session(GtkWidget *widget, char *cfgpath, char *globcfgpath, GroachData *groach_data)
{
	properties_save(groach_data->groach_prop, cfgpath);

	return FALSE;
}

static void
run_button_cb(GtkButton *button, GroachData *groach_data)
{
	if (gro_controller_is_running(groach_data->controller) == FALSE) {
		gro_controller_run(groach_data->controller);
	} else {
		gro_controller_add_gmoves(groach_data->controller, NUM_GMOVES_ADD);
	}
}

static void
stop_button_cb(GtkButton *button, GroachData *groach_data)
{
	if (gro_controller_is_running(groach_data->controller) == TRUE) {
		gro_controller_stop(groach_data->controller);
	}
}

static void
speed_changed_cb(GtkAdjustment *adj, GroachData *groach_data)
{
	gro_controller_change_interval(groach_data->controller, adj->value);

	if (gro_controller_is_running(groach_data->controller) == TRUE) {
		gro_controller_stop(groach_data->controller);
		gro_controller_run(groach_data->controller);
	}
}

static void
about_cb(AppletWidget *widget, gpointer data)
{
	static GtkWidget *about = NULL;
	static const gchar *authors[] = {
		"INOUE Seiichiro <inoue@ainet.or.jp>",
		NULL
	};

	if (about) {
		gdk_window_raise(GTK_WIDGET(about)->window);
	} else {
		about = gnome_about_new(_("Groach applet"), VERSION,
								_("Copyright 1999, under the GNU General Public License."),
								authors,
								/* another comments */
								_("This applet has no use what-so-ever, too. "
								  "It only takes up disk space and "
								  "compilation time, and if loaded it also "
								  "takes up precious panel space and memory. "
								  "If anyone is found using this applet, he "
								  "should be promptly sent for a psychiatric "
								  "evaluation."),
								NULL);
		gtk_signal_connect(GTK_OBJECT(about), "destroy",
						   GTK_SIGNAL_FUNC(gtk_widget_destroyed), &about);
		gtk_widget_show(about);
	}
	return;
}


/**
 * sighup_handler:
 * SIGHUP signal is sent by myself, when all roaches are dead.
 **/
static void
sighup_handler(int sig)
{
	return;/* do nothing now */
}
