//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Cube.hpp,v 1.5 2004/07/11 14:11:14 delpinux Exp $

// This class provides a toolkit to manipulate Cubes. It inherits from the
// Shape class.

#ifndef CUBE_HPP
#define CUBE_HPP

#include <Shape.hpp>
#include <Index.hpp>
/*!
  \class Cube
  This is the class which defines a Virtual Reality Cube.

  \author Stphane Del Pino.
*/
class Cube
  : public Shape
{
private:
  /*! Two oposite corners of the box:
    \li \a lower is the corner whose coordinates
    are the smallest (one by one)
    \li \a higher is the corner whose coordinates
    are the highest (one by one)
  */
  Vertex __lower, __higher; 

  //! returns true if the point \a p is inside the shape.
  inline bool inShape(const TinyVector<3, real_t>& p) const;

  //! Prints the Cube informations using the stream \a s.
  std::ostream& put(std::ostream& s) const;

public:
  const real_t& lowerCorner(const size_t& i) const
  {
    return __lower[i];
  }

  const real_t& higherCorner(const size_t& i) const
  {
    return __higher[i];
  }

  //! Copies the Cube \a C.
  const Cube& operator = (const Cube& C);

  //! Builds a Cube based on vertices \a a and \a b
  Cube(const Vertex& a, const Vertex& b);

  //! Default constructor.
  Cube();

  //! Copy constructor.
  Cube(const Cube& C);

  //! Destructor.
  ~Cube()
  {
    ;
  }
};

//! Returns true if the point \a p is inside the Cube.
inline bool Cube::inShape(const TinyVector<3, real_t>& p) const
{
  return ((p[0] <= __higher.x()) and
	  (p[1] <= __higher.y()) and
	  (p[2] <= __higher.z()) and
	  (p[0] >= __lower.x())  and
	  (p[1] >= __lower.y())  and
	  (p[2] >= __lower.z()));
}

#endif // CUBE_HPP

