/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: shortcut.cxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 14:43:14 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/
#include <unistd.h>
#include <sys/stat.h>
#include <rtl/ustring.hxx>

#include <errcode.hxx>
#include <shortcut.hxx>
#include <stream.hxx>

static const char* unicode_section = ".W";

using namespace rtl;

static String aEmptyString;

#if OSL_DEBUG_LEVEL > 1
void UnxShortcut::printGroups()
{
	fprintf( stderr, "\nUnxShortcut for file %s\n", m_aFilename.GetStr() );
	for( int i = 0; i < m_aGroups.Count(); i++ )
	{
		ShortcutGroup* pGroup = m_aGroups.GetObject( i );
		fprintf( stderr, "Group %d (%s)\n", i, pGroup->getName().GetStr() );
		pGroup->writeEntries( stderr, TRUE );
	}
}
#endif

ShortcutGroup::~ShortcutGroup()
{
	while( m_aEntries.Count() )
		delete m_aEntries.Remove( (ULONG)0 );
}

const String& ShortcutGroup::getValue( const String& rKey ) const
{
	ShortcutEntry* pEntry;
	for( int i = 0; i < m_aEntries.Count(); i++ )
	{
		pEntry = m_aEntries.GetObject( i );
		if( pEntry->aName == rKey )
			return pEntry->aValue;
	}
	return aEmptyString;
}

void ShortcutGroup::setValue( const String& rKey, const String& rValue )
{
	ShortcutEntry* pEntry;
	for( int i = 0; i < m_aEntries.Count(); i++ )
	{
		pEntry = m_aEntries.GetObject( i );
		if( pEntry->aName == rKey )
		{
			pEntry->aValue = rValue;
			return;
		}
	}
	m_aEntries.Insert( new ShortcutEntry( rKey, rValue ), LIST_APPEND );
}

const String& ShortcutGroup::getKeyName( UINT32 nIndex ) const
{
	if( nIndex < m_aEntries.Count() )
		return m_aEntries.GetObject( nIndex)->aName;
	return aEmptyString;
}

void ShortcutGroup::writeEntries( FILE* fp, BOOL bUnixLineEnds ) const
{
	char *pEnd = bUnixLineEnds ? "\n" : "\r\n";

	ShortcutEntry* pEntry;
	for( int i = 0; i < m_aEntries.Count(); i++ )
	{
		pEntry = m_aEntries.GetObject( i );
		OString aValue;
		if( m_aGroupname.Search( unicode_section ) == m_aGroupname.Len()-2 )
		{
			OUString aUValue = OStringToOUString( OString( pEntry->aValue.GetStr() ), GetSystemCharSet() );
			aValue = OUStringToOString( aUValue, RTL_TEXTENCODING_UTF7 );
		}
		else
			aValue = pEntry->aValue.GetStr();

		fprintf( fp, "%s=%s%s",
				 pEntry->aName.GetStr(), aValue.getStr(),
				 pEnd );
	}
	fprintf( fp, pEnd );
}

UnxShortcut::UnxShortcut( const ItemIDPath& rFolderPath,
						  const String rQualifier) :
		m_aFolderPath( rFolderPath ),
		m_aQualifier( rQualifier ),
		m_pQualGroup( NULL )
{
	if( m_aQualifier.Len() )
		m_aQualifier += unicode_section;

	m_aGroups.Insert( new ShortcutGroup( "" ), LIST_APPEND );
	if( m_aQualifier.Len() )
	{
		m_aGroups.Insert( new ShortcutGroup( m_aQualifier ) );
		m_pQualGroup = m_aGroups.GetObject( 1 );
	}
	else
		m_pQualGroup = m_aGroups.GetObject( 0 );
}

UnxShortcut::~UnxShortcut()
{
	while( m_aGroups.Count() )
		delete m_aGroups.Remove( (ULONG)0 );
}

UINT32 UnxShortcut::GetValueNameCount() const
{
	::osl::Guard< ::osl::Mutex > aGuard( const_cast<UnxShortcut*>(this)->m_aMutex );

	return m_aGroups.GetObject( 0 )->getKeyCount();
}

const String& UnxShortcut::GetValueName( UINT32 nIndex ) const
{
	::osl::Guard< ::osl::Mutex > aGuard( const_cast<UnxShortcut*>(this)->m_aMutex );

	return m_aGroups.GetObject( 0 )->getKeyName( nIndex );
}

const String& UnxShortcut::GetValueContent( const String& crValueName ) const
{
	::osl::Guard< ::osl::Mutex > aGuard( const_cast<UnxShortcut*>(this)->m_aMutex );

	const String& rRet = m_pQualGroup->getValue( crValueName );
	return rRet.Len() ? rRet : m_aGroups.GetObject( 0 )->getValue( crValueName );
}

void UnxShortcut::SetValueContent( const String& crValueName,
								   const String& crValueContent )
{
	::osl::Guard< ::osl::Mutex > aGuard( m_aMutex );

	m_pQualGroup->setValue( crValueName, crValueContent );
	// overwrite default value every time
	m_aGroups.GetObject( 0 )->setValue( crValueName, crValueContent );
}

FSysError UnxShortcut::Load( const ItemIDPath &crIDPath )
{
	::osl::Guard< ::osl::Mutex > aGuard( m_aMutex );

	ItemIDPath aPath( m_aFolderPath );
	aPath += crIDPath;
	m_aFilename = aPath.GetHostNotationPath();

	struct stat aStat;
	if( lstat( m_aFilename.GetStr(), &aStat ) )
		return FSYS_ERR_NOTEXISTS;

	if( ! S_ISREG( aStat.st_mode ) && ! S_ISLNK( aStat.st_mode ) )
		return FSYS_ERR_NOTAFILE;

	char buf[65536];

	if( S_ISLNK( aStat.st_mode ) )
	{
		SetValueContent( "Title", m_aFilename.GetToken( m_aFilename.GetTokenCount( '/' ) - 1, '/' ) );
		int nChars;
		if( ( nChars = readlink( m_aFilename.GetStr(), buf, sizeof(buf) ) ) >= 0 )
		{
			buf [ nChars ] = 0;
			String aUrl( buf );
			if( aUrl.Search( '/' ) == STRING_NOTFOUND )
			{
				// prepend path
				aUrl = m_aFilename;
				aUrl.SetToken( aUrl.GetTokenCount( '/' ) - 1,
							   '/',
							   buf );
			}
			else if( aUrl.GetChar( 0 ) == '.' )
			{
				if( aUrl.GetChar( 1 ) == '/' || aUrl.GetChar( 1 ) == '.' )
				{
					aUrl = m_aFilename;
					aUrl.SetToken( aUrl.GetTokenCount( '/' ) - 1,
								   '/',
								   "" );
					aUrl += '/';
					aUrl += buf;
				}
			}
			aUrl = "file://" + aUrl;
			SetValueContent( "URL", aUrl );
		}
		m_eOldFormat = SHORTCUT_FORMAT_SYSTEM;
		return FSYS_ERR_OK;
	}

	while( m_aGroups.Count() )
		delete m_aGroups.Remove( (ULONG)0 );
	m_aGroups.Insert( new ShortcutGroup( "" ), LIST_APPEND );
	m_pQualGroup = m_aGroups.GetObject( 0 );

	SvFileStream aInput( m_aFilename, STREAM_READ );
	if( ! aInput.IsOpen() )
		return FSYS_ERR_ACCESSDENIED;

	String aLine;
	ShortcutGroup* pCurrentGroup = m_aGroups.GetObject( 0 );
	int nPos;
	while( ! aInput.IsEof() )
	{
		aInput.ReadLine( aLine );
		while( aLine[ 0 ] == ' ' || aLine[ 0 ] == '\t' || aLine[ 0 ] == '\r' )
			aLine.Erase( 0, 1 );

		if( aLine[0] == '[' && ( nPos = aLine.Search( ']' ) ) != STRING_NOTFOUND )
		{
			String aGroup( aLine.Copy( 1, nPos - 1 ) );
			if( aGroup.Compare( "InternetShortcut", 16 ) == COMPARE_EQUAL )
			{
				// strip extra - from Store method if necessary
				aGroup.Erase( 0, aGroup.Len() < 17 ? 16 : 17 );

				m_eOldFormat = SHORTCUT_FORMAT_URL;
			}
			else if( aGroup.Compare( "KDE Desktop Entry", 17 ) == COMPARE_EQUAL )
			{
				aGroup.Erase( 0, 17 );
				m_eOldFormat = SHORTCUT_FORMAT_BESTFIT;
			}
			if( ! aGroup.Len() )
				pCurrentGroup = m_aGroups.GetObject( 0 );
			else
			{
				pCurrentGroup = NULL;
				for( int i = 0; i < m_aGroups.Count(); i++ )
				{
					if( m_aGroups.GetObject( i )->getName() == aGroup )
					{
						pCurrentGroup = m_aGroups.GetObject( i );
						break;
					}
				}
				if( pCurrentGroup == NULL )
				{
					pCurrentGroup = new ShortcutGroup( aGroup );
					m_aGroups.Insert( pCurrentGroup, LIST_APPEND );
				}
			}
			if( aGroup == m_aQualifier )
				m_pQualGroup = pCurrentGroup;
		}
		else if( ( nPos = aLine.Search( '=' ) ) != STRING_NOTFOUND )
		{
			String aName( aLine.Copy( 0, nPos ) );
			String aValue( aLine.Copy( nPos+1 ) );
			if( pCurrentGroup->getName().Search( unicode_section ) == pCurrentGroup->getName().Len()-2 )
			{
				OUString aUValue = OStringToOUString( OString( aValue.GetStr() ), RTL_TEXTENCODING_UTF7 );
				OString aOValue = OUStringToOString( aUValue, GetSystemCharSet() );
				aValue = aOValue.getStr();
			}
			pCurrentGroup->setValue( aName, aValue );
			if( ! m_aGroups.GetObject( 0 )->getValue( aName ).Len() )
				m_aGroups.GetObject( 0 )->setValue( aName, aValue );
		}
	}

	if( ! m_pQualGroup->getName().Len() )
	{
		m_pQualGroup = new ShortcutGroup( m_aQualifier );
		m_aGroups.Insert( m_pQualGroup, LIST_APPEND );
	}
	return FSYS_ERR_OK;
}

FSysError UnxShortcut::Store( const String & crTitle,
							  ItemIDPath & crNewIDPath,
							  ShortcutFormat ePreferred )
{
	::osl::Guard< ::osl::Mutex > aGuard( m_aMutex );

	if( crTitle.Len() )
	{
		ItemIDPath aPath( m_aFolderPath );
		crNewIDPath = aPath;

		m_aFilename = aPath.GetHostNotationPath();
		m_aFilename += "/";
		m_aFilename += crTitle;
	}
	else
		ePreferred = m_eOldFormat;

	switch( ePreferred )
	{
		case SHORTCUT_FORMAT_URL:
		{
			if( crTitle.Len() )
				m_aFilename += ".url";

			FILE* fp = fopen( m_aFilename.GetStr(), "w" );
			if( ! fp )
				return FSYS_ERR_ACCESSDENIED;
			fprintf( fp, "%s\r\n", GetValueContent( "URL" ).GetStr() );
			for( int i = 0; i < m_aGroups.Count(); i++ )
			{
				ShortcutGroup* pGroup = m_aGroups.GetObject( i );
				String aGroup( "InternetShortcut" );
				if( pGroup->getName().Len() )
				{
					aGroup += '-';
					aGroup += pGroup->getName();
				}
				fprintf( fp, "[%s]\r\n", aGroup.GetStr() );
				pGroup->writeEntries( fp );
			}

			fclose( fp );
		}
		break;
		case SHORTCUT_FORMAT_BESTFIT:
		{
			if( pDtIntegrator_GetBookmarksDir &&
				pDtIntegrator_GetBookmarksDir().
				Search( ".kde/share/apps/kfm/bookmarks" ) != STRING_NOTFOUND )
			{
				if( crTitle.Len() )
					m_aFilename += ".kdelnk";
				FILE* fp = fopen( m_aFilename.GetStr(), "w" );
				if( ! fp )
					return FSYS_ERR_ACCESSDENIED;

				ShortcutGroup* pDefaultGroup = m_aGroups.GetObject( 0 );

				String aURL = pDefaultGroup->getValue( "URL" );
				String aNewURL( aURL );
				aNewURL.SearchAndReplace( "file:///", "file:/" );
				pDefaultGroup->setValue( "URL", aNewURL );
				fprintf( fp, "# KDE Config File\n" );
				fprintf( fp, "[KDE Desktop Entry]\n" );
				fprintf( fp, "Type=Link\n" );
				pDefaultGroup->writeEntries( fp, TRUE );
				pDefaultGroup->setValue( "URL", aURL );
				fclose( fp );
				break;
			}
		}
		case SHORTCUT_FORMAT_SYSTEM:
		{
			ItemIDPath aToPath( GetValueContent( "URL" ) );
			symlink( aToPath.GetHostNotationPath().GetStr(), m_aFilename.GetStr() );
		}
		break;
	}

	crNewIDPath += m_aFilename.GetToken( m_aFilename.GetTokenCount( '/' )-1, '/' );

	return FSYS_ERR_OK;
}
