/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: BlockManager.java,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 16:20:32 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

package com.sun.xmlsearch.db;

import java.io.*;

/* TODO
   _nBlocksLimit shouldn't be hardwired
*/

final class BlockManager {
    private final class BlockDescriptor {
		public Block   _block = null;
		public boolean _lock = false;
		public boolean _modf = false;
		public int     _prev = 0;
		public int     _next = 0;
  
		public void reset() {
			_lock = _modf = false;
			_block = null;
		}
    } // end of BlockDescriptor

    private final static int INCR = 64; // size increment
    private final RandomAccessFile _file;
    private final long             _blockSize;
    private final boolean          _update;
    private final BlockFactory     _blockFactory;
    private final Block            _dummy;
    private final int              _nBlocksLimit = 64;
    private int                    _blockTableSize;
    private BlockDescriptor[]      _blockTab;
    private int                    _nBlocks = 0;
    private int                    _oldest = 0;
    private int                    _newest = 0;
  
    public BlockManager(BlockManagerParameters params,
						final boolean update,
						BlockFactory bfactory)
		throws IOException {
		_blockFactory = bfactory;
		_update = update;
		//    params.readState();
		_blockSize = (long)params.getBlockSize();
		_file = new RandomAccessFile(params.getFile(), update ? "rw" : "r");
		_dummy = bfactory.makeBlock();
		if (_file.length() == 0 && update) {
			_dummy.setBlockNumber(0);
			writeBlock(_dummy);
		}
		_blockTableSize = (int)(_file.length()/_blockSize);
		_blockTab = new BlockDescriptor[_blockTableSize];
		mapBlock(0, new BlockDescriptor(), bfactory.makeBlock());
    }
  
    public void lockBlock(int blNum) {
		_blockTab[blNum]._lock = true;
    }

    public void unlockBlock(int blNum) {
		_blockTab[blNum]._lock = false;
    }

    public void setModified(int blNum) {
		_blockTab[blNum]._modf = true;
    }

    public void close() throws IOException {
		if (_update)
			for (int i = 0; i < _blockTableSize; i++)
				if (_blockTab[i] != null && _blockTab[i]._modf)
					writeBlock(_blockTab[i]._block);
		_file.close();
    }

    public Block accessBlock(int blockNumber) throws Exception {
		if (_blockTab[blockNumber] != null)
			moveToFront(blockNumber);
		else if (_nBlocks < _nBlocksLimit)
			mapBlock(blockNumber, new BlockDescriptor(),
					 _blockFactory.makeBlock());
		else
			remapSomeBlock(blockNumber);
		return _blockTab[blockNumber]._block;
    }
  
    public Block getNewBlock() throws Exception {
		final int number = (int)(_file.length()/_blockSize);
		if (number > _blockTableSize - 1) {
			BlockDescriptor[] newArray =
				new BlockDescriptor[_blockTableSize + INCR];
			System.arraycopy(_blockTab, 0, newArray, 0, _blockTableSize);
			_blockTab = newArray;
			_blockTableSize += INCR;
		}
		if (_nBlocks < _nBlocksLimit) {
			final Block bl = _blockFactory.makeBlock();
			bl.setBlockNumber(number);
			writeBlock(bl);
			addDescriptor(bl, number, new BlockDescriptor());
		}
		else {
			_dummy.setBlockNumber(number);
			writeBlock(_dummy);
			remapSomeBlock(number);
		}
		return _blockTab[number]._block;
    }

    private void mapBlock(int blockNumber, BlockDescriptor desc, Block block)
		throws IOException {
		_file.seek(_blockSize * blockNumber);
		Block.readIn(_file, block);
		addDescriptor(block, blockNumber, desc);
    }
  
    private void addDescriptor(Block block, int blockNumber,
							   BlockDescriptor desc) {
		_blockTab[blockNumber] = desc;
		desc._block = block;
		_blockTab[desc._prev = _newest]._next = blockNumber;
		_newest = blockNumber;
		_nBlocks++;
    }
  
    private void remapSomeBlock(int blockNumber) throws Exception {
		int index = _oldest;
		while (_blockTab[index]._lock && index != _newest)
			index = _blockTab[index]._next;
		if (_blockTab[index]._lock)
			throw new Exception("everything locked");
		if (_blockTab[index]._modf)
			writeBlock(_blockTab[index]._block);
    
		_nBlocks--;
		Block reused = _blockTab[index]._block;
		// delete from double-linked list
		if (index == _oldest)
			_oldest = _blockTab[index]._next;
		else if (index == _newest)
			_newest = _blockTab[index]._prev;
		else {
			_blockTab[_blockTab[index]._next]._prev = _blockTab[index]._prev;
			_blockTab[_blockTab[index]._prev]._next = _blockTab[index]._next;
		}
		_blockTab[index].reset();
		mapBlock(blockNumber, _blockTab[index], reused);
		//System.err.println("reuse "+index+" --> "+blockNumber+" "+_nBlocks);
		_blockTab[index] = null;
    }

    private void moveToFront(int index) {
		if (index == _oldest) {
			_oldest = _blockTab[index]._next;
			_blockTab[index]._prev = _newest;
			_blockTab[_newest]._next = index;
			_newest = index;
		}
		else if (index != _newest) {
			_blockTab[_blockTab[index]._next]._prev = _blockTab[index]._prev;
			_blockTab[_blockTab[index]._prev]._next = _blockTab[index]._next;
			_blockTab[index]._prev = _newest;
			_blockTab[_newest]._next = index;
			_newest = index;
		}
    }
  
    public void writeBlock(Block bl) throws IOException {
		_file.seek(_blockSize * bl._number);
		bl.writeOut(_file);
    }

    public void mapBlocks(BlockProcessor processor) throws IOException {
		long nBlocks = _file.length()/_blockSize;
		Block block = _blockFactory.makeBlock();
		_file.seek(0);
		for (int i = 0; i < nBlocks; i++)
			processor.process(Block.readIn(_file, block));
    }

    /**
	 * Debug code
	 */

    private boolean debug=false;
    private void debug(String msg) {
		if (debug) {
			System.err.println("Block Manager: "+msg);
		}
    }
}
